/* Test driver to read and decode an encoded ASN.1 message 
 * This file provides sample code for processing the PBX response
 *
 * @Author Avin Patel
 */

#include <stdio.h>
#include <stdlib.h>
#include "client.h"


int Client::receiveCSTAMessage(OSRTSocket& cstaSocket)
{
   ASN1BERDecodeBuffer decodeBuffer;
   if(receiveMessage(cstaSocket, decodeBuffer) < 0) {
      printf("error: failed to receive message\n");
      return -1;
   }

   ASN1T_CSTA_ROSE_PDU pdu;
   ASN1C_CSTA_ROSE_PDU pduC (decodeBuffer, pdu);

   int stat = pduC.Decode ();

   if (stat == 0) {
      if (trace) {
         printf ("Decode of ROSE PDU was successful\n");
         pduC.Print ("pdu");
      }
   }
   else {
      printf ("Decode of ROSE PDU failed\n");
      decodeBuffer.printErrorInfo ();
      return -1;
   }

   /* Process the ROSE operation */
   processPDU (pdu, trace);
   printf ("***\n");

   return 0;
}

void Client::processPDU (ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   if (trace) {
      printf ("processing Message ..\n" );
   }

   switch(pdu.t){
      case T_CSTA_ROSE_PDU_invoke:
      processInvoke(pdu, trace);
      break;
     
      case T_CSTA_ROSE_PDU_returnResult:
      processReturnResult(pdu, trace);
      break;

      case T_CSTA_ROSE_PDU_returnError:
      processReturnError(pdu, trace);
      break;
       
      case T_CSTA_ROSE_PDU_reject:
      processReject(pdu, trace);
      break;

      default: break;
   }
}

void Client::processInvoke(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /*
   ASN1T_CSTA_ROSE_PDU_invoke.argument(opentype) needs to be decoded/encoded
   using ASN1T_CSTA_ROSE_PDU_invoke.opcode value as defined in Table 1.
   */

   if(pdu.u.invoke->opcode.t != T_Code_local) {
      printf ("ROSE Operation's Invoke.parameter type is not known\n");
      return;
   }
   
   ASN1BERDecodeBuffer decodeBuffer2 
      (pdu.u.invoke->argument.data, pdu.u.invoke->argument.numocts);
   int stat;
   switch(pdu.u.invoke->opcode.u.local) {
      case 1:   /* alternateCall operation Request */
      {
      ASN1T_AlternateCallArgument alternateCallArg;
      ASN1C_AlternateCallArgument alternateCallArgC(decodeBuffer2, alternateCallArg);
      stat = alternateCallArgC.Decode();
      if (stat == 0) {
         /* process the "alternateCall" operation Request in your application */
         if (trace) {
           printf ("Decode of AlternateCallArgument was successful\n");
           alternateCallArgC.Print ("AlternateCallArgument");
         }
      }
      else {
         printf ("Decode of AlternateCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 2:   /* answerCall operation Request */
      {
      ASN1T_AnswerCallArgument answerCallArg;
      ASN1C_AnswerCallArgument answerCallArgC(decodeBuffer2, answerCallArg);
      stat = answerCallArgC.Decode();
      if (stat == 0) {
         /* process the "answerCall" operation Request in your application */
         if (trace) {
           printf ("Decode of AnswerCallArgument was successful\n");
           answerCallArgC.Print ("AnswerCallArgument");
         }
      }
      else {
         printf ("Decode of AnswerCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 3:   /* callCompletion operation Request */
      {
      ASN1T_CallCompletionArgument callCompletionArg;
      ASN1C_CallCompletionArgument callCompletionArgC(decodeBuffer2, callCompletionArg);
      stat = callCompletionArgC.Decode();
      if (stat == 0) {
         /* process the "callCompletion" operation Request in your application */
         if (trace) {
           printf ("Decode of CallCompletionArgument was successful\n");
           callCompletionArgC.Print ("CallCompletionArgument");
         }
      }
      else {
         printf ("Decode of CallCompletionArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 4:   /* clearCall operation Request */
      {
      ASN1T_ClearCallArgument clearCallArg;
      ASN1C_ClearCallArgument clearCallArgC(decodeBuffer2, clearCallArg);
      stat = clearCallArgC.Decode();
      if (stat == 0) {
         /* process the "clearCall" operation Request in your application */
         if (trace) {
           printf ("Decode of ClearCallArgument was successful\n");
           clearCallArgC.Print ("ClearCallArgument");
         }
      }
      else {
         printf ("Decode of ClearCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 5:   /* clearConnection operation Request */
      {
      ASN1T_ClearConnectionArgument clearConnectionArg;
      ASN1C_ClearConnectionArgument clearConnectionArgC(decodeBuffer2, clearConnectionArg);
      stat = clearConnectionArgC.Decode();
      if (stat == 0) {
         /* process the "clearConnection" operation Request in your application */
         if (trace) {
           printf ("Decode of ClearConnectionArgument was successful\n");
           clearConnectionArgC.Print ("ClearConnectionArgument");
         }
      }
      else {
         printf ("Decode of ClearConnectionArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 6:   /* conferenceCall operation Request */
      {
      ASN1T_ConferenceCallArgument conferenceCallArg;
      ASN1C_ConferenceCallArgument conferenceCallArgC(decodeBuffer2, conferenceCallArg);
      stat = conferenceCallArgC.Decode();
      if (stat == 0) {
         /* process the "conferenceCall" operation Request in your application */
         if (trace) {
           printf ("Decode of ConferenceCallArgument was successful\n");
           conferenceCallArgC.Print ("ConferenceCallArgument");
         }
      }
      else {
         printf ("Decode of ConferenceCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 7:   /* consultationCall operation Request */
      {
      ASN1T_ConsultationCallArgument consultationCallArg;
      ASN1C_ConsultationCallArgument consultationCallArgC(decodeBuffer2, consultationCallArg);
      stat = consultationCallArgC.Decode();
      if (stat == 0) {
         /* process the "consultationCall" operation Request in your application */
         if (trace) {
           printf ("Decode of ConsultationCallArgument was successful\n");
           consultationCallArgC.Print ("ConsultationCallArgument");
         }
      }
      else {
         printf ("Decode of ConsultationCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 8:   /* divertCall operation Request */
      {
      ASN1T_DivertCallArgument divertCallArg;
      ASN1C_DivertCallArgument divertCallArgC(decodeBuffer2, divertCallArg);
      stat = divertCallArgC.Decode();
      if (stat == 0) {
         /* process the "divertCall" operation Request in your application */
         if (trace) {
           printf ("Decode of DivertCallArgument was successful\n");
           divertCallArgC.Print ("DivertCallArgument");
         }
      }
      else {
         printf ("Decode of DivertCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 9:   /* holdCall operation Request */
      {
      ASN1T_HoldCallArgument holdCallArg;
      ASN1C_HoldCallArgument holdCallArgC(decodeBuffer2, holdCallArg);
      stat = holdCallArgC.Decode();
      if (stat == 0) {
         /* process the "holdCall" operation Request in your application */
         if (trace) {
           printf ("Decode of HoldCallArgument was successful\n");
           holdCallArgC.Print ("HoldCallArgument");
         }
      }
      else {
         printf ("Decode of HoldCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 10:   /* makeCall operation Request */
      {
      ASN1T_MakeCallArgument makeCallArg;
      ASN1C_MakeCallArgument makeCallArgC(decodeBuffer2, makeCallArg);
      stat = makeCallArgC.Decode();
      if (stat == 0) {
         /* process the "makeCall" operation Request in your application */
         if (trace) {
           printf ("Decode of MakeCallArgument was successful\n");
           makeCallArgC.Print ("MakeCallArgument");
         }
      }
      else {
         printf ("Decode of MakeCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 11:   /* makePredictiveCall operation Request */
      {
      ASN1T_MakePredictiveCallArgument makePredictiveCallArg;
      ASN1C_MakePredictiveCallArgument makePredictiveCallArgC(decodeBuffer2, makePredictiveCallArg);
      stat = makePredictiveCallArgC.Decode();
      if (stat == 0) {
         /* process the "makePredictiveCall" operation Request in your application */
         if (trace) {
           printf ("Decode of MakePredictiveCallArgument was successful\n");
           makePredictiveCallArgC.Print ("MakePredictiveCallArgument");
         }
      }
      else {
         printf ("Decode of MakePredictiveCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 12:   /* queryDevice operation Request */
      {
      ASN1T_QueryDeviceArgument queryDeviceArg;
      ASN1C_QueryDeviceArgument queryDeviceArgC(decodeBuffer2, queryDeviceArg);
      stat = queryDeviceArgC.Decode();
      if (stat == 0) {
         /* process the "queryDevice" operation Request in your application */
         if (trace) {
           printf ("Decode of QueryDeviceArgument was successful\n");
           queryDeviceArgC.Print ("QueryDeviceArgument");
         }
      }
      else {
         printf ("Decode of QueryDeviceArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 13:   /* reconnectCall operation Request */
      {
      ASN1T_ReconnectCallArgument reconnectCallArg;
      ASN1C_ReconnectCallArgument reconnectCallArgC(decodeBuffer2, reconnectCallArg);
      stat = reconnectCallArgC.Decode();
      if (stat == 0) {
         /* process the "reconnectCall" operation Request in your application */
         if (trace) {
           printf ("Decode of ReconnectCallArgument was successful\n");
           reconnectCallArgC.Print ("ReconnectCallArgument");
         }
      }
      else {
         printf ("Decode of ReconnectCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 14:   /* retrieveCall operation Request */
      {
      ASN1T_RetrieveCallArgument retrieveCallArg;
      ASN1C_RetrieveCallArgument retrieveCallArgC(decodeBuffer2, retrieveCallArg);
      stat = retrieveCallArgC.Decode();
      if (stat == 0) {
         /* process the "retrieveCall" operation Request in your application */
         if (trace) {
           printf ("Decode of RetrieveCallArgument was successful\n");
           retrieveCallArgC.Print ("RetrieveCallArgument");
         }
      }
      else {
         printf ("Decode of RetrieveCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 15:   /* setFeature operation Request */
      {
      ASN1T_SetFeatureArgument setFeatureArg;
      ASN1C_SetFeatureArgument setFeatureArgC(decodeBuffer2, setFeatureArg);
      stat = setFeatureArgC.Decode();
      if (stat == 0) {
         /* process the "setFeature" operation Request in your application */
         if (trace) {
           printf ("Decode of SetFeatureArgument was successful\n");
           setFeatureArgC.Print ("SetFeatureArgument");
         }
      }
      else {
         printf ("Decode of SetFeatureArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 16:   /* transferCall operation Request */
      {
      ASN1T_TransferCallArgument transferCallArg;
      ASN1C_TransferCallArgument transferCallArgC(decodeBuffer2, transferCallArg);
      stat = transferCallArgC.Decode();
      if (stat == 0) {
         /* process the "transferCall" operation Request in your application */
         if (trace) {
           printf ("Decode of TransferCallArgument was successful\n");
           transferCallArgC.Print ("TransferCallArgument");
         }
      }
      else {
         printf ("Decode of TransferCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 21:   /* cSTAEventReport operation Request */
      {
      ASN1T_CSTAEventReportArgument cSTAEventReportArg;
      ASN1C_CSTAEventReportArgument cSTAEventReportArgC(decodeBuffer2, cSTAEventReportArg);
      stat = cSTAEventReportArgC.Decode();
      if (stat == 0) {
         if (trace) {
           printf ("Decode of CSTAEventReportArgument was successful\n");
           cSTAEventReportArgC.Print ("CSTAEventReportArgument");
         }
         /* process the "cSTAEventReport" operation Request in your application */
         processEvent(cSTAEventReportArg, trace);
      }
      else {
         printf ("Decode of CSTAEventReportArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 31:   /* routeRequest operation Request */
      {
      ASN1T_RouteRequestArgument routeRequestArg;
      ASN1C_RouteRequestArgument routeRequestArgC(decodeBuffer2, routeRequestArg);
      stat = routeRequestArgC.Decode();
      if (stat == 0) {
         /* process the "routeRequest" operation Request in your application */
         if (trace) {
           printf ("Decode of RouteRequestArgument was successful\n");
           routeRequestArgC.Print ("RouteRequestArgument");
         }
      }
      else {
         printf ("Decode of RouteRequestArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 32:   /* reRouteRequest operation Request */
      {
      ASN1T_ReRouteRequestArgument reRouteRequestArg;
      ASN1C_ReRouteRequestArgument reRouteRequestArgC(decodeBuffer2, reRouteRequestArg);
      stat = reRouteRequestArgC.Decode();
      if (stat == 0) {
         /* process the "reRouteRequest" operation Request in your application */
         if (trace) {
           printf ("Decode of ReRouteRequestArgument was successful\n");
           reRouteRequestArgC.Print ("ReRouteRequestArgument");
         }
      }
      else {
         printf ("Decode of ReRouteRequestArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 33:   /* routeSelectRequest operation Request */
      {
      ASN1T_RouteSelectRequestArgument routeSelectRequestArg;
      ASN1C_RouteSelectRequestArgument routeSelectRequestArgC(decodeBuffer2, routeSelectRequestArg);
      stat = routeSelectRequestArgC.Decode();
      if (stat == 0) {
         /* process the "routeSelectRequest" operation Request in your application */
         if (trace) {
           printf ("Decode of RouteSelectRequestArgument was successful\n");
           routeSelectRequestArgC.Print ("RouteSelectRequestArgument");
         }
      }
      else {
         printf ("Decode of RouteSelectRequestArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 34:   /* routeUsedRequest operation Request */
      {
      ASN1T_RouteUsedRequestArgument routeUsedRequestArg;
      ASN1C_RouteUsedRequestArgument routeUsedRequestArgC(decodeBuffer2, routeUsedRequestArg);
      stat = routeUsedRequestArgC.Decode();
      if (stat == 0) {
         /* process the "routeUsedRequest" operation Request in your application */
         if (trace) {
           printf ("Decode of RouteUsedRequestArgument was successful\n");
           routeUsedRequestArgC.Print ("RouteUsedRequestArgument");
         }
      }
      else {
         printf ("Decode of RouteUsedRequestArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 35:   /* routeEndRequest operation Request */
      {
      ASN1T_RouteEndRequestArgument routeEndRequestArg;
      ASN1C_RouteEndRequestArgument routeEndRequestArgC(decodeBuffer2, routeEndRequestArg);
      stat = routeEndRequestArgC.Decode();
      if (stat == 0) {
         /* process the "routeEndRequest" operation Request in your application */
         if (trace) {
           printf ("Decode of RouteEndRequestArgument was successful\n");
           routeEndRequestArgC.Print ("RouteEndRequestArgument");
         }
      }
      else {
         printf ("Decode of RouteEndRequestArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 51:   /* escapeService operation Request */
      {
      ASN1T_EscapeServiceArgument escapeServiceArg;
      ASN1C_EscapeServiceArgument escapeServiceArgC(decodeBuffer2, escapeServiceArg);
      stat = escapeServiceArgC.Decode();
      if (stat == 0) {
         /* process the "escapeService" operation Request in your application */
         if (trace) {
           printf ("Decode of EscapeServiceArgument was successful\n");
           escapeServiceArgC.Print ("EscapeServiceArgument");
         }
      }
      else {
         printf ("Decode of EscapeServiceArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 52:   /* systemStatus operation Request */
      {
      ASN1T_SystemStatusArgument systemStatusArg;
      ASN1C_SystemStatusArgument systemStatusArgC(decodeBuffer2, systemStatusArg);
      stat = systemStatusArgC.Decode();
      if (stat == 0) {
         processSystemStatusReq(systemStatusArg);
         if (trace) {
           printf ("Decode of SystemStatusArgument was successful\n");
           systemStatusArgC.Print ("SystemStatusArgument");
         }
      }
      else {
         printf ("Decode of SystemStatusArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 71:   /* monitorStart operation Request */
      {
      ASN1T_MonitorStartArgument monitorStartArg;
      ASN1C_MonitorStartArgument monitorStartArgC(decodeBuffer2, monitorStartArg);
      stat = monitorStartArgC.Decode();
      if (stat == 0) {
         /* process the "monitorStart" operation Request in your application */
         if (trace) {
            printf ("Decode of MonitorStartArgument was successful\n");
            monitorStartArgC.Print ("MonitorStartArgument");
         }
      }
      else {
         printf ("Decode of MonitorStartArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 72:   /* changeMonitorFilter operation Request */
      {
      ASN1T_ChangeMonitorFilterArgument changeMonitorFilterArg;
      ASN1C_ChangeMonitorFilterArgument changeMonitorFilterArgC(decodeBuffer2, changeMonitorFilterArg);
      stat = changeMonitorFilterArgC.Decode();
      if (stat == 0) {
         /* process the "changeMonitorFilter" operation Request in your application */
         if (trace) {
           printf ("Decode of ChangeMonitorFilterArgument was successful\n");
           changeMonitorFilterArgC.Print ("ChangeMonitorFilterArgument");
         }
      }
      else {
         printf ("Decode of ChangeMonitorFilterArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 73:   /* monitorStop operation Request */
      {
      ASN1T_MonitorStopArgument monitorStopArg;
      ASN1C_MonitorStopArgument monitorStopArgC(decodeBuffer2, monitorStopArg);
      stat = monitorStopArgC.Decode();
      if (stat == 0) {
         /* process the "monitorStop" operation Request in your application */
         if (trace) {
           printf ("Decode of MonitorStopArgument was successful\n");
           monitorStopArgC.Print ("MonitorStopArgument");
         }
      }
      else {
         printf ("Decode of MonitorStopArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 74:   /* snapshotDevice operation Request */
      {
      ASN1T_SnapshotDeviceArgument snapshotDeviceArg;
      ASN1C_SnapshotDeviceArgument snapshotDeviceArgC(decodeBuffer2, snapshotDeviceArg);
      stat = snapshotDeviceArgC.Decode();
      if (stat == 0) {
         /* process the "snapshotDevice" operation Request in your application */
         if (trace) {
           printf ("Decode of SnapshotDeviceArgument was successful\n");
           snapshotDeviceArgC.Print ("SnapshotDeviceArgument");
         }
      }
      else {
         printf ("Decode of SnapshotDeviceArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 75:   /* snapshotCall operation Request */
      {
      ASN1T_SnapshotCallArgument snapshotCallArg;
      ASN1C_SnapshotCallArgument snapshotCallArgC(decodeBuffer2, snapshotCallArg);
      stat = snapshotCallArgC.Decode();
      if (stat == 0) {
         /* process the "snapshotCall" operation Request in your application */
         if (trace) {
           printf ("Decode of SnapshotCallArgument was successful\n");
           snapshotCallArgC.Print ("SnapshotCallArgument");
         }
      }
      else {
         printf ("Decode of SnapshotCallArgument failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

     default:
      printf ("error: invalid operation code, not defined in csta phase 3\n");
      break;
   }
}

void Client::processEvent(ASN1T_CSTAEventReportArgument& eventReportArg, 
   OSBOOL trace)
{

   if(eventReportArg.eventType.t != T_EventTypeID_cSTAform) {
     printf ("EVENT's CSTAEventReportArgument.eventInfo type is not known\n");
     return;
   }

   ASN1BERDecodeBuffer decodeBuffer2 (eventReportArg.eventInfo.data, 
     eventReportArg.eventInfo.numocts);
   int stat;

   {
   switch(eventReportArg.eventType.u.cSTAform) {
   case 1: {  /* callCleared event */
      ASN1T_CallClearedEventInfo callClearedInfo;
      ASN1C_CallClearedEventInfo callClearedInfoC(decodeBuffer2, callClearedInfo);
      stat = callClearedInfoC.Decode();
      if (stat == 0) {
         /* process the "callCleared" event in your application */
         if (trace) {
           printf ("Decode of CallClearedEventInfo was successful\n");
           callClearedInfoC.Print ("CallClearedEventInfo");
         }
      }
      else {
         printf ("Decode of CallClearedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 2: {  /* conferenced event */
      ASN1T_ConferencedEventInfo conferencedInfo;
      ASN1C_ConferencedEventInfo conferencedInfoC(decodeBuffer2, conferencedInfo);
      stat = conferencedInfoC.Decode();
      if (stat == 0) {
         /* process the "conferenced" event in your application */
         if (trace) {
           printf ("Decode of ConferencedEventInfo was successful\n");
           conferencedInfoC.Print ("ConferencedEventInfo");
         }
      }
      else {
         printf ("Decode of ConferencedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 3: {  /* connectionCleared event */
      ASN1T_ConnectionClearedEventInfo connectionClearedInfo;
      ASN1C_ConnectionClearedEventInfo connectionClearedInfoC(decodeBuffer2, connectionClearedInfo);
      stat = connectionClearedInfoC.Decode();
      if (stat == 0) {
         /* process the "connectionCleared" event in your application */
         if (trace) {
           printf ("Decode of ConnectionClearedEventInfo was successful\n");
           connectionClearedInfoC.Print ("ConnectionClearedEventInfo");
         }
      }
      else {
         printf ("Decode of ConnectionClearedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 4: {  /* delivered event */
      ASN1T_DeliveredEventInfo deliveredInfo;
      ASN1C_DeliveredEventInfo deliveredInfoC(decodeBuffer2, deliveredInfo);
      stat = deliveredInfoC.Decode();
      if (stat == 0) {
         /* process the "delivered" event in your application */
         if (trace) {
           printf ("Decode of DeliveredEventInfo was successful\n");
           deliveredInfoC.Print ("DeliveredEventInfo");
         }
      }
      else {
         printf ("Decode of DeliveredEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 5: {  /* diverted event */
      ASN1T_DivertedEventInfo divertedInfo;
      ASN1C_DivertedEventInfo divertedInfoC(decodeBuffer2, divertedInfo);
      stat = divertedInfoC.Decode();
      if (stat == 0) {
         /* process the "diverted" event in your application */
         if (trace) {
           printf ("Decode of DivertedEventInfo was successful\n");
           divertedInfoC.Print ("DivertedEventInfo");
         }
      }
      else {
         printf ("Decode of DivertedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 6: {  /* established event */
      ASN1T_EstablishedEventInfo establishedInfo;
      ASN1C_EstablishedEventInfo establishedInfoC(decodeBuffer2, establishedInfo);
      stat = establishedInfoC.Decode();
      if (stat == 0) {
         /* process the "established" event in your application */
         if (trace) {
           printf ("Decode of EstablishedEventInfo was successful\n");
           establishedInfoC.Print ("EstablishedEventInfo");
         }
      }
      else {
         printf ("Decode of EstablishedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 7: {  /* failed event */
      ASN1T_FailedEventInfo failedInfo;
      ASN1C_FailedEventInfo failedInfoC(decodeBuffer2, failedInfo);
      stat = failedInfoC.Decode();
      if (stat == 0) {
         /* process the "failed" event in your application */
         if (trace) {
           printf ("Decode of FailedEventInfo was successful\n");
           failedInfoC.Print ("FailedEventInfo");
         }
      }
      else {
         printf ("Decode of FailedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 8: {  /* held event */
      ASN1T_HeldEventInfo heldInfo;
      ASN1C_HeldEventInfo heldInfoC(decodeBuffer2, heldInfo);
      stat = heldInfoC.Decode();
      if (stat == 0) {
         /* process the "held" event in your application */
         if (trace) {
           printf ("Decode of HeldEventInfo was successful\n");
           heldInfoC.Print ("HeldEventInfo");
         }
      }
      else {
         printf ("Decode of HeldEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 9: {  /* networkReached event */
      ASN1T_NetworkReachedEventInfo networkReachedInfo;
      ASN1C_NetworkReachedEventInfo networkReachedInfoC(decodeBuffer2, networkReachedInfo);
      stat = networkReachedInfoC.Decode();
      if (stat == 0) {
         /* process the "networkReached" event in your application */
         if (trace) {
           printf ("Decode of NetworkReachedEventInfo was successful\n");
           networkReachedInfoC.Print ("NetworkReachedEventInfo");
         }
      }
      else {
         printf ("Decode of NetworkReachedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 10: {  /* originated event */
      ASN1T_OriginatedEventInfo originatedInfo;
      ASN1C_OriginatedEventInfo originatedInfoC(decodeBuffer2, originatedInfo);
      stat = originatedInfoC.Decode();
      if (stat == 0) {
         /* process the "originated" event in your application */
         if (trace) {
           printf ("Decode of OriginatedEventInfo was successful\n");
           originatedInfoC.Print ("OriginatedEventInfo");
         }
      }
      else {
         printf ("Decode of OriginatedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 11: {  /* queued event */
      ASN1T_QueuedEventInfo queuedInfo;
      ASN1C_QueuedEventInfo queuedInfoC(decodeBuffer2, queuedInfo);
      stat = queuedInfoC.Decode(); 
      if (stat == 0) {
         /* process the "queued" event in your application */
         if (trace) {
           printf ("Decode of QueuedEventInfo was successful\n");
           queuedInfoC.Print ("QueuedEventInfo");
         }
      }
      else {
         printf ("Decode of QueuedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 12: {  /* retrieved event */
      ASN1T_RetrievedEventInfo retrievedInfo;
      ASN1C_RetrievedEventInfo retrievedInfoC(decodeBuffer2, retrievedInfo);
      stat = retrievedInfoC.Decode();
      if (stat == 0) {
         /* process the "retrieved" event in your application */
         if (trace) {
           printf ("Decode of RetrievedEventInfo was successful\n");
           retrievedInfoC.Print ("RetrievedEventInfo");
         }
      }
      else {
         printf ("Decode of RetrievedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 13: {  /* serviceInitiated event */
      ASN1T_ServiceInitiatedEventInfo serviceInitiatedInfo;
      ASN1C_ServiceInitiatedEventInfo serviceInitiatedInfoC(decodeBuffer2, serviceInitiatedInfo);
      stat = serviceInitiatedInfoC.Decode();
      if (stat == 0) {
         /* process the "serviceInitiated" event in your application */
         if (trace) {
           printf ("Decode of ServiceInitiatedEventInfo was successful\n");
           serviceInitiatedInfoC.Print ("ServiceInitiatedEventInfo");
         }
      }
      else {
         printf ("Decode of ServiceInitiatedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 14: {  /* transferred event */
      ASN1T_TransferedEventInfo transferredInfo;
      ASN1C_TransferedEventInfo transferredInfoC(decodeBuffer2, transferredInfo);
      stat = transferredInfoC.Decode();
      if (stat == 0) {
         /* process the "transferred" event in your application */
         if (trace) {
           printf ("Decode of TransferedEventInfo was successful\n");
           transferredInfoC.Print ("TransferedEventInfo");
         }
      }
      else {
         printf ("Decode of TransferedEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 101: {  /* callInformation event */
      ASN1T_CallInformationEventInfo callInformationInfo;
      ASN1C_CallInformationEventInfo callInformationInfoC(decodeBuffer2, callInformationInfo);
      stat = callInformationInfoC.Decode();
      if (stat == 0) {
         /* process the "callInformation" event in your application */
         if (trace) {
           printf ("Decode of CallInformationEventInfo was successful\n");
           callInformationInfoC.Print ("CallInformationEventInfo");
         }
      }
      else {
         printf ("Decode of CallInformationEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 102: {  /* doNotDisturb event */
      ASN1T_DoNotDisturbEventInfo doNotDisturbInfo;
      ASN1C_DoNotDisturbEventInfo doNotDisturbInfoC(decodeBuffer2, doNotDisturbInfo);
      stat = doNotDisturbInfoC.Decode();
      if (stat == 0) {
         /* process the "doNotDisturb" event in your application */
         if (trace) {
           printf ("Decode of DoNotDisturbEventInfo was successful\n");
           doNotDisturbInfoC.Print ("DoNotDisturbEventInfo");
         }
      }
      else {
         printf ("Decode of DoNotDisturbEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 103: {  /* forwarding event */
      ASN1T_ForwardingEventInfo forwardingInfo;
      ASN1C_ForwardingEventInfo forwardingInfoC(decodeBuffer2, forwardingInfo);
      stat = forwardingInfoC.Decode();
      if (stat == 0) {
         /* process the "forwarding" event in your application */
         if (trace) {
           printf ("Decode of ForwardingEventInfo was successful\n");
           forwardingInfoC.Print ("ForwardingEventInfo");
         }
      }
      else {
         printf ("Decode of ForwardingEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 104: {  /* messageWaiting event */
      ASN1T_MessageWaitingEventInfo messageWaitingInfo;
      ASN1C_MessageWaitingEventInfo messageWaitingInfoC(decodeBuffer2, messageWaitingInfo);
      stat = messageWaitingInfoC.Decode();
      if (stat == 0) {
         /* process the "messageWaiting" event in your application */
         if (trace) {
           printf ("Decode of MessageWaitingEventInfo was successful\n");
           messageWaitingInfoC.Print ("MessageWaitingEventInfo");
         }
      }
      else {
         printf ("Decode of MessageWaitingEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 201: {  /* loggedOn event */
      ASN1T_LoggedOnEventInfo loggedOnInfo;
      ASN1C_LoggedOnEventInfo loggedOnInfoC(decodeBuffer2, loggedOnInfo);
      stat = loggedOnInfoC.Decode();
      if (stat == 0) {
         /* process the "loggedOn" event in your application */
         if (trace) {
           printf ("Decode of LoggedOnEventInfo was successful\n");
           loggedOnInfoC.Print ("LoggedOnEventInfo");
         }
      }
      else {
         printf ("Decode of LoggedOnEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 202: {  /* loggedOff event */
      ASN1T_LoggedOffEventInfo loggedOffInfo;
      ASN1C_LoggedOffEventInfo loggedOffInfoC(decodeBuffer2, loggedOffInfo);
      stat = loggedOffInfoC.Decode();
      if (stat == 0) {
         /* process the "loggedOff" event in your application */
         if (trace) {
           printf ("Decode of LoggedOffEventInfo was successful\n");
           loggedOffInfoC.Print ("LoggedOffEventInfo");
         }
      }
      else {
         printf ("Decode of LoggedOffEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 203: {  /* notReady event */
      ASN1T_NotReadyEventInfo notReadyInfo;
      ASN1C_NotReadyEventInfo notReadyInfoC(decodeBuffer2, notReadyInfo);
      stat = notReadyInfoC.Decode();
      if (stat == 0) {
         /* process the "notReady" event in your application */
         if (trace) {
           printf ("Decode of NotReadyEventInfo was successful\n");
           notReadyInfoC.Print ("NotReadyEventInfo");
         }
      }
      else {
         printf ("Decode of NotReadyEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 204: {  /* ready event */
      ASN1T_ReadyEventInfo readyInfo;
      ASN1C_ReadyEventInfo readyInfoC(decodeBuffer2, readyInfo);
      stat = readyInfoC.Decode();
      if (stat == 0) {
         /* process the "ready" event in your application */
         if (trace) {
           printf ("Decode of ReadyEventInfo was successful\n");
           readyInfoC.Print ("ReadyEventInfo");
         }
      }
      else {
         printf ("Decode of ReadyEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 205: {  /* workNotReady event */
      ASN1T_WorkNotReadyEventInfo workNotReadyInfo;
      ASN1C_WorkNotReadyEventInfo workNotReadyInfoC(decodeBuffer2, workNotReadyInfo);
      stat = workNotReadyInfoC.Decode();
      if (stat == 0) {
         /* process the "workNotReady" event in your application */
         if (trace) {
           printf ("Decode of WorkNotReadyEventInfo was successful\n");
           workNotReadyInfoC.Print ("WorkNotReadyEventInfo");
         }
      }
      else {
         printf ("Decode of WorkNotReadyEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 206: {  /* workReady event */
      ASN1T_WorkReadyEventInfo workReadyInfo;
      ASN1C_WorkReadyEventInfo workReadyInfoC(decodeBuffer2, workReadyInfo);
      stat = workReadyInfoC.Decode();
      if (stat == 0) {
         /* process the "workReady" event in your application */
         if (trace) {
           printf ("Decode of WorkReadyEventInfo was successful\n");
           workReadyInfoC.Print ("WorkReadyEventInfo");
         }
      }
      else {
         printf ("Decode of WorkReadyEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 301: {  /* backInService event */
      ASN1T_BackInServiceEventInfo backInServiceInfo;
      ASN1C_BackInServiceEventInfo backInServiceInfoC(decodeBuffer2, backInServiceInfo);
      stat = backInServiceInfoC.Decode();
      if (stat == 0) {
         /* process the "backInService" event in your application */
         if (trace) {
           printf ("Decode of BackInServiceEventInfo was successful\n");
           backInServiceInfoC.Print ("BackInServiceEventInfo");
         }
      }
      else {
         printf ("Decode of BackInServiceEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 302: {  /* outOfService event */
      ASN1T_OutOfServiceEventInfo outOfServiceInfo;
      ASN1C_OutOfServiceEventInfo outOfServiceInfoC(decodeBuffer2, outOfServiceInfo);
      stat = outOfServiceInfoC.Decode();
      if (stat == 0) {
         /* process the "outOfService" event in your application */
         if (trace) {
           printf ("Decode of OutOfServiceEventInfo was successful\n");
           outOfServiceInfoC.Print ("OutOfServiceEventInfo");
         }
      }
      else {
         printf ("Decode of OutOfServiceEventInfo failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
   }break;
   case 401: {  /* private event */
      /* NULL type */
   }break;
   default:
      printf ("error: invalid operation code, not defined in csta phase 3\n");
      break;
  }
}
}

void Client::processReturnResult(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /*
   ASN1T_ReturnResult.result.result(opentype) needs to be decoded
   using ASN1T_ReturnResult.result.opcode value as defined in Table 1.
   */

   if(pdu.u.returnResult->result.opcode.t != T_Code_local) {
     printf ("ROSE Operation's ReturnResult.result.result type is not known\n");
     return;
   }

   ASN1BERDecodeBuffer decodeBuffer2 (pdu.u.returnResult->result.result.data, 
     pdu.u.returnResult->result.result.numocts);
   int stat;

   switch(pdu.u.returnResult->result.opcode.u.local) {
      case 1 :  /* alternateCall operation Response*/
      {
      ASN1T_AlternateCallResult alternateCallRes;
      ASN1C_AlternateCallResult alternateCallResC(decodeBuffer2, alternateCallRes);
      stat = alternateCallResC.Decode();
      if (stat == 0) {
         /* process the "alternateCall" operation response in your application */
         if (trace) {
           printf ("Decode of AlternateCallResult was successful\n");
           alternateCallResC.Print ("AlternateCallResult");
         }
      }
      else {
         printf ("Decode of AlternateCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 2 :  /* answerCall operation Response*/
      {
      ASN1T_AnswerCallResult answerCallRes;
      ASN1C_AnswerCallResult answerCallResC(decodeBuffer2, answerCallRes);
      stat = answerCallResC.Decode();
      if (stat == 0) {
         processAnswerCallRes(answerCallRes);
         if (trace) {
           printf ("Decode of AnswerCallResult was successful\n");
           answerCallResC.Print ("AnswerCallResult");
         }
      }
      else {
         printf ("Decode of AnswerCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 3 :  /* callCompletion operation Response*/
      {
      ASN1T_CallCompletionResult callCompletionRes;
      ASN1C_CallCompletionResult callCompletionResC(decodeBuffer2, callCompletionRes);
      stat = callCompletionResC.Decode();
      if (stat == 0) {
         /* process the "callCompletion" operation response in your application */
         if (trace) {
           printf ("Decode of CallCompletionResult was successful\n");
           callCompletionResC.Print ("CallCompletionResult");
         }
      }
      else {
         printf ("Decode of CallCompletionResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 4 :  /* clearCall operation Response*/
      {
      ASN1T_ClearCallResult clearCallRes;
      ASN1C_ClearCallResult clearCallResC(decodeBuffer2, clearCallRes);
      stat = clearCallResC.Decode();
      if (stat == 0) {
         /* process the "clearCall" operation response in your application */
         if (trace) {
           printf ("Decode of ClearCallResult was successful\n");
           clearCallResC.Print ("ClearCallResult");
         }
      }
      else {
         printf ("Decode of ClearCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 5 :  /* clearConnection operation Response*/
      {
      ASN1T_ClearConnectionResult clearConnectionRes;
      ASN1C_ClearConnectionResult clearConnectionResC(decodeBuffer2, clearConnectionRes);
      stat = clearConnectionResC.Decode();
      if (stat == 0) {
         /* process the "clearConnection" operation response in your application */
         if (trace) {
           printf ("Decode of ClearConnectionResult was successful\n");
           clearConnectionResC.Print ("ClearConnectionResult");
         }
      }
      else {
         printf ("Decode of ClearConnectionResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 6 :  /* conferenceCall operation Response*/
      {
      ASN1T_ConferenceCallResult conferenceCallRes;
      ASN1C_ConferenceCallResult conferenceCallResC(decodeBuffer2, conferenceCallRes);
      stat = conferenceCallResC.Decode();
      if (stat == 0) {
         /* process the "conferenceCall" operation response in your application */
         if (trace) {
           printf ("Decode of ConferenceCallResult was successful\n");
           conferenceCallResC.Print ("ConferenceCallResult");
         }
      }
      else {
         printf ("Decode of ConferenceCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 7 :  /* consultationCall operation Response*/
      {
      ASN1T_ConsultationCallResult consultationCallRes;
      ASN1C_ConsultationCallResult consultationCallResC(decodeBuffer2, consultationCallRes);
      stat = consultationCallResC.Decode();
      if (stat == 0) {
         /* process the "consultationCall" operation response in your application */
         if (trace) {
           printf ("Decode of ConsultationCallResult was successful\n");
           consultationCallResC.Print ("ConsultationCallResult");
         }
      }
      else {
         printf ("Decode of ConsultationCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 8 :  /* divertCall operation Response*/
      {
      ASN1T_DivertCallResult divertCallRes;
      ASN1C_DivertCallResult divertCallResC(decodeBuffer2, divertCallRes);
      stat = divertCallResC.Decode();
      if (stat == 0) {
         processDivertCallRes(divertCallRes);
         if (trace) {
           printf ("Decode of DivertCallResult was successful\n");
           divertCallResC.Print ("DivertCallResult");
         }
      }
      else {
         printf ("Decode of DivertCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 9 :  /* holdCall operation Response*/
      {
      ASN1T_HoldCallResult holdCallRes;
      ASN1C_HoldCallResult holdCallResC(decodeBuffer2, holdCallRes);
      stat = holdCallResC.Decode();
      if (stat == 0) {
         /* process the "holdCall" operation response in your application */
         if (trace) {
           printf ("Decode of HoldCallResult was successful\n");
           holdCallResC.Print ("HoldCallResult");
         }
      }
      else {
         printf ("Decode of HoldCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 10 :  /* makeCall operation Response*/
      {
      ASN1T_MakeCallResult makeCallRes;
      ASN1C_MakeCallResult makeCallResC(decodeBuffer2, makeCallRes);
      stat = makeCallResC.Decode();
      if (stat == 0) {
         processMakeCallResult(makeCallRes);
         if (trace) {
           printf ("Decode of MakeCallResult was successful\n");
           makeCallResC.Print ("MakeCallResult");
         }
      }
      else {
         printf ("Decode of MakeCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 11 :  /* makePredictiveCall operation Response*/
      {
      ASN1T_MakePredictiveCallResult makePredictiveCallRes;
      ASN1C_MakePredictiveCallResult makePredictiveCallResC(decodeBuffer2, makePredictiveCallRes);
      stat = makePredictiveCallResC.Decode();
      if (stat == 0) {
         /* process the "makePredictiveCall" operation response in your application */
         if (trace) {
           printf ("Decode of MakePredictiveCallResult was successful\n");
           makePredictiveCallResC.Print ("MakePredictiveCallResult");
         }
      }
      else {
         printf ("Decode of MakePredictiveCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 12 :  /* queryDevice operation Response*/
      {
      ASN1T_QueryDeviceResult queryDeviceRes;
      ASN1C_QueryDeviceResult queryDeviceResC(decodeBuffer2, queryDeviceRes);
      stat = queryDeviceResC.Decode();
      if (stat == 0) {
         processQueryDeviceRes(queryDeviceRes);
         if (trace) {
           printf ("Decode of QueryDeviceResult was successful\n");
           queryDeviceResC.Print ("QueryDeviceResult");
         }
      }
      else {
         printf ("Decode of QueryDeviceResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 13 :  /* reconnectCall operation Response*/
      {
      ASN1T_ReconnectCallResult reconnectCallRes;
      ASN1C_ReconnectCallResult reconnectCallResC(decodeBuffer2, reconnectCallRes);
      stat = reconnectCallResC.Decode();
      if (stat == 0) {
         /* process the "reconnectCall" operation response in your application */
         if (trace) {
           printf ("Decode of ReconnectCallResult was successful\n");
           reconnectCallResC.Print ("ReconnectCallResult");
         }
      }
      else {
         printf ("Decode of ReconnectCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 14 :  /* retrieveCall operation Response*/
      {
      ASN1T_RetrieveCallResult retrieveCallRes;
      ASN1C_RetrieveCallResult retrieveCallResC(decodeBuffer2, retrieveCallRes);
      stat = retrieveCallResC.Decode();
      if (stat == 0) {
         /* process the "retrieveCall" operation response in your application */
         if (trace) {
           printf ("Decode of RetrieveCallResult was successful\n");
           retrieveCallResC.Print ("RetrieveCallResult");
         }
      }
      else {
         printf ("Decode of RetrieveCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 15 :  /* setFeature operation Response*/
      {
      ASN1T_SetFeatureResult setFeatureRes;
      ASN1C_SetFeatureResult setFeatureResC(decodeBuffer2, setFeatureRes);
      stat = setFeatureResC.Decode();
      if (stat == 0) {
         /* process the "setFeature" operation response in your application */
         if (trace) {
           printf ("Decode of SetFeatureResult was successful\n");
           setFeatureResC.Print ("SetFeatureResult");
         }
      }
      else {
         printf ("Decode of SetFeatureResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 16 :  /* transferCall operation Response*/
      {
      ASN1T_TransferCallResult transferCallRes;
      ASN1C_TransferCallResult transferCallResC(decodeBuffer2, transferCallRes);
      stat = transferCallResC.Decode();
      if (stat == 0) {
         processTransferCallRes(transferCallRes);
         if (trace) {
           printf ("Decode of TransferCallResult was successful\n");
           transferCallResC.Print ("TransferCallResult");
         }
      }
      else {
         printf ("Decode of TransferCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 51 :  /* escapeService operation Response*/
      {
      ASN1T_EscapeServiceResult escapeServiceRes;
      ASN1C_EscapeServiceResult escapeServiceResC(decodeBuffer2, escapeServiceRes);
      stat = escapeServiceResC.Decode();
      if (stat == 0) {
         /* process the "escapeService" operation response in your application */
         if (trace) {
           printf ("Decode of EscapeServiceResult was successful\n");
           escapeServiceResC.Print ("EscapeServiceResult");
         }
      }
      else {
         printf ("Decode of EscapeServiceResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 52 :  /* systemStatus operation Response*/
      {
      ASN1T_SystemStatusResult systemStatusRes;
      ASN1C_SystemStatusResult systemStatusResC(decodeBuffer2, systemStatusRes);
      stat = systemStatusResC.Decode();
      if (stat == 0) {
         /* process the "systemStatus" operation response in your application */
         if (trace) {
           printf ("Decode of SystemStatusResult was successful\n");
           systemStatusResC.Print ("SystemStatusResult");
         }
      }
      else {
         printf ("Decode of SystemStatusResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 71 :  /* monitorStart operation Response*/
      {
      ASN1T_MonitorStartResult monitorStartRes;
      ASN1C_MonitorStartResult monitorStartResC(decodeBuffer2, monitorStartRes);
      stat = monitorStartResC.Decode();
      if (stat == 0) {
         processMonitorStartRes(monitorStartRes);
         if (trace) {
            printf ("Decode of MonitorStartResult was successful\n");
            monitorStartResC.Print ("MonitorStartResult");
         }
      }
      else {
         printf ("Decode of MonitorStartResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 72 :  /* changeMonitorFilter operation Response*/
      {
      ASN1T_ChangeMonitorFilterResult changeMonitorFilterRes;
      ASN1C_ChangeMonitorFilterResult changeMonitorFilterResC(decodeBuffer2, changeMonitorFilterRes);
      stat = changeMonitorFilterResC.Decode();
      if (stat == 0) {
         /* process the "changeMonitorFilter" operation response in your application */
         if (trace) {
           printf ("Decode of ChangeMonitorFilterResult was successful\n");
           changeMonitorFilterResC.Print ("ChangeMonitorFilterResult");
         }
      }
      else {
         printf ("Decode of ChangeMonitorFilterResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 73 :  /* monitorStop operation Response*/
      {
      ASN1T_MonitorStopResult monitorStopRes;
      ASN1C_MonitorStopResult monitorStopResC(decodeBuffer2, monitorStopRes);
      stat = monitorStopResC.Decode();
      if (stat == 0) {
         processMonitorStopRes(monitorStopRes);
         if (trace) {
           printf ("Decode of MonitorStopResult was successful\n");
           monitorStopResC.Print ("MonitorStopResult");
         }
      }
      else {
         printf ("Decode of MonitorStopResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 74 :  /* snapshotDevice operation Response*/
      {
      ASN1T_SnapshotDeviceResult snapshotDeviceRes;
      ASN1C_SnapshotDeviceResult snapshotDeviceResC(decodeBuffer2, snapshotDeviceRes);
      stat = snapshotDeviceResC.Decode();
      if (stat == 0) {
         /* process the "snapshotDevice" operation response in your application */
         if (trace) {
           printf ("Decode of SnapshotDeviceResult was successful\n");
           snapshotDeviceResC.Print ("SnapshotDeviceResult");
         }
      }
      else {
         printf ("Decode of SnapshotDeviceResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

      case 75 :  /* snapshotCall operation Response*/
      {
      ASN1T_SnapshotCallResult snapshotCallRes;
      ASN1C_SnapshotCallResult snapshotCallResC(decodeBuffer2, snapshotCallRes);
      stat = snapshotCallResC.Decode();
      if (stat == 0) {
         /* process the "snapshotCall" operation response in your application */
         if (trace) {
           printf ("Decode of SnapshotCallResult was successful\n");
           snapshotCallResC.Print ("SnapshotCallResult");
         }
      }
      else {
         printf ("Decode of SnapshotCallResult failed\n");
         decodeBuffer2.printErrorInfo ();
         return;
      }
      }
      break;

     default: /* error: invalid operation code, not defined in csta phase 3*/
      printf ("error: invalid operation code, not defined in csta phase 3\n");
      break;
   }
}

void Client::processReturnError(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /* 
   return error processing will require to decode the ASN1T_ReturnError.parameter
   opentype in to real error type, only one possible error type can be returned
   in csta 1, which is "UniversalFailure" 
   
   ASN1T_ReturnError.errcode field is used to findout the real type for 
   ASN1T_ReturnError.parameter(opentype). Real type for all the operation
   in Table 1(i.e csta1) is "UniversalFailure"
   */
   int stat;
   ASN1BERDecodeBuffer decodeBuffer2 
     (pdu.u.returnError->parameter.data, pdu.u.returnError->parameter.numocts);
   ASN1T_UniversalFailure uniFailure;
   ASN1C_UniversalFailure uniFailureC (decodeBuffer2, uniFailure);
   stat = uniFailureC.Decode ();
   if (stat == 0) {
     /* process the ReturnError in your application */
      if (trace) {
         printf ("Decode of ReturnError was successful\n");
         uniFailureC.Print ("UniversalFailure");
      }
   }
   else {
      printf ("Decode of ReturnError failed\n");
      decodeBuffer2.printErrorInfo ();
      return ;
   }
}

void Client::processReject(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /* process the reject in your application */
}

