/* Test driver to read and decode an encoded ASN.1 message 
 * This file provides sample code for processing the PBX response
 *
 * @Author Avin Patel
 */

#include <stdio.h>
#include <stdlib.h>
#include "CSTA-ROSE-PDU-types.h"
#include "CSTA-alternate-call.h"
#include "CSTA-answer-call.h"
#include "CSTA-associate-data.h"
#include "CSTA-call-completion.h"
#include "CSTA-clear-call.h"
#include "CSTA-clear-connection.h"
#include "CSTA-conference-call.h"
#include "CSTA-consultation-call.h"
#include "CSTA-divert-call.h"
#include "CSTA-hold-call.h"
#include "CSTA-make-call.h"
#include "CSTA-make-predictive-call.h"
#include "CSTA-park-call.h"
#include "CSTA-query-device.h"
#include "CSTA-reconnect-call.h"
#include "CSTA-retrieve-call.h"
#include "CSTA-send-DTMF-tones.h"
#include "CSTA-set-feature.h"
#include "CSTA-single-step-conference.h"
#include "CSTA-single-step-transfer.h"
#include "CSTA-transfer-call.h"
#include "CSTA-event-report-definitions.h"
#include "CSTA-call-cleared-event.h"
#include "CSTA-conferenced-event.h"
#include "CSTA-connection-cleared-event.h"
#include "CSTA-delivered-event.h"
#include "CSTA-diverted-event.h"
#include "CSTA-established-event.h"
#include "CSTA-failed-event.h"
#include "CSTA-held-event.h"
#include "CSTA-network-reached-event.h"
#include "CSTA-originated-event.h"
#include "CSTA-queued-event.h"
#include "CSTA-retrieved-event.h"
#include "CSTA-service-initiated-event.h"
#include "CSTA-transferred-event.h"
#include "CSTA-auto-answer-event.h"
#include "CSTA-call-information-event.h"
#include "CSTA-do-not-disturb-event.h"
#include "CSTA-forwarding-event.h"
#include "CSTA-message-waiting-event.h"
#include "CSTA-microphone-mute-event.h"
#include "CSTA-speaker-mute-event.h"
#include "CSTA-speaker-volume-event.h"
#include "CSTA-agent-busy-event.h"
#include "CSTA-logged-on-event.h"
#include "CSTA-logged-off-event.h"
#include "CSTA-not-ready-event.h"
#include "CSTA-ready-event.h"
#include "CSTA-working-after-call-event.h"
#include "CSTA-back-in-service-event.h"
#include "CSTA-out-of-service-event.h"
#include "CSTA-private-event.h"
#include "CSTA-play-event.h"
#include "CSTA-record-event.h"
#include "CSTA-review-event.h"
#include "CSTA-stop-event.h"
#include "CSTA-suspend-play-event.h"
#include "CSTA-suspend-record-event.h"
#include "CSTA-voice-attributes-change-event.h"
#include "CSTA-route-request.h"
#include "CSTA-re-route-request.h"
#include "CSTA-Route-select-request.h"
#include "CSTA-route-used-request.h"
#include "CSTA-route-end-request.h"
#include "CSTA-escape-service.h"
#include "CSTA-system-status.h"
#include "CSTA-monitor-start.h"
#include "CSTA-change-monitor-filter.h"
#include "CSTA-monitor-stop.h"
#include "CSTA-snapshot-device.h"
#include "CSTA-snapshot-call.h"
#include "CSTA-start-data-path.h"
#include "CSTA-stop-data-path.h"
#include "CSTA-send-data.h"
#include "CSTA-send-multicast-data.h"
#include "CSTA-send-broadcast-data.h"
#include "CSTA-suspend-data-path.h"
#include "CSTA-data-path-suspended.h"
#include "CSTA-resume-data-path.h"
#include "CSTA-data-path-resumed.h"
#include "CSTA-fast-data.h"
#include "CSTA-concatenate-message.h"
#include "CSTA-delete-message.h"
#include "CSTA-play-message.h"
#include "CSTA-query-voice-attribute.h"
#include "CSTA-record-message.h"
#include "CSTA-reposition.h"
#include "CSTA-resume.h"
#include "CSTA-review.h"
#include "CSTA-set-voice-attribute.h"
#include "CSTA-stop.h"
#include "CSTA-suspend.h"
#include "CSTA-synthesize-message.h"
#include "CSTA-error-definition.h"
#include "CSTA-event-causes.h"
#include "CSTA-switching-function-objects.h"
#include "CSTA-device-identifiers.h"
#include "CSTA-call-connection-identifiers.h"
#include "CSTA-connection-states.h"
#include "CSTA-status-reporting.h"
#include "CSTA-device-feature-types.h"
#include "CSTA-security.h"
#include "CSTA-extension-types.h"
#include "CSTA-application-context-information.h"
#include "Remote-Operations-Generic-ROS-PDUs.h"
#include "Remote-Operations-Information-Objects.h"
#include "Remote-Operations-Useful-Definitions.h"
#include "client.h"

int Client::receiveCSTAMessage(OSRTSocket& cstaSocket)
{
   ASN1BERDecodeBuffer decodeBuffer;
   if(receiveMessage(cstaSocket, decodeBuffer) < 0) {
      printf("error: failed to receive message\n");
      return -1;
   }

   ASN1T_CSTA_ROSE_PDU pdu;
   ASN1C_CSTA_ROSE_PDU pduC (decodeBuffer, pdu);

   int stat = pduC.Decode ();

   if (stat == 0) {
      if (trace) {
         printf ("Decode of ROSE PDU was successful\n");
         pduC.Print ("pdu");
      }
   }
   else {
      printf ("Decode of ROSE PDU failed\n");
      decodeBuffer.printErrorInfo ();
      return -1;
   }

   /* Process the ROSE operation */
   processPDU (pdu, trace);
   printf ("***\n");

   return 0;
}

void Client::processPDU(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   switch(pdu.t){
      case T_CSTA_ROSE_PDU_invoke:
         processInvoke(pdu, trace);
         break;
      
      case T_CSTA_ROSE_PDU_returnResult:
         processReturnResult(pdu, trace);
         break;

      case T_CSTA_ROSE_PDU_returnError:
         processReturnError(pdu, trace);
         break;
         
      case T_CSTA_ROSE_PDU_reject:
         processReject(pdu, trace);
         break;

      default: break;
   }
}

void Client::processInvoke(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /*
   ASN1T_CSTA_ROSE_PDU_invoke.argument(opentype) needs to be decoded/encoded
   using ASN1T_CSTA_ROSE_PDU_invoke.opcode value as defined in Table 1.
   */

   if(pdu.u.invoke->opcode.t != T_Code_local) {
      printf ("ROSE Operation's Invoke.parameter type is not known\n");
      return;
   }
   
   ASN1BERDecodeBuffer decodeBuffer2 
      (pdu.u.invoke->argument.data, pdu.u.invoke->argument.numocts);
   int stat;
   switch(pdu.u.invoke->opcode.u.local) {
      case 1:   /* alternateCall operation Request */
         {
         ASN1T_AlternateCallArgument alternateCallArg;
         ASN1C_AlternateCallArgument alternateCallArgC(decodeBuffer2, alternateCallArg);
         stat = alternateCallArgC.Decode();
         if (stat == 0) {
            /* process the "alternateCall" operation Request in your application */
            if (trace) {
               printf ("Decode of AlternateCallArgument was successful\n");
               alternateCallArgC.Print ("AlternateCallArgument");
            }
         }
         else {
            printf ("Decode of AlternateCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
         }
         break;

      case 2:   /* answerCall operation Request */
         {
         ASN1T_AnswerCallArgument answerCallArg;
         ASN1C_AnswerCallArgument answerCallArgC(decodeBuffer2, answerCallArg);
         stat = answerCallArgC.Decode();
         if (stat == 0) {
            /* process the "answerCall" operation Request in your application */
            if (trace) {
               printf ("Decode of AnswerCallArgument was successful\n");
               answerCallArgC.Print ("AnswerCallArgument");
            }
         }
         else {
            printf ("Decode of AnswerCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 17:   /* associateData operation Request */
         {
         ASN1T_CSTA_associate_data_AssociateDataArgument associateDataArg;
         ASN1C_CSTA_associate_data_AssociateDataArgument associateDataArgC(decodeBuffer2, associateDataArg);
         stat = associateDataArgC.Decode();
         if (stat == 0) {
            /* process the "associateData" operation Request in your application */
            if (trace) {
               printf ("Decode of AssociateDataArgument was successful\n");
               associateDataArgC.Print ("AssociateDataArgument");
            }
         }
         else {
            printf ("Decode of AssociateDataArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 3:   /* callCompletion operation Request */
         {
         ASN1T_CallCompletionArgument callCompletionArg;
         ASN1C_CallCompletionArgument callCompletionArgC(decodeBuffer2, callCompletionArg);
         stat = callCompletionArgC.Decode();
         if (stat == 0) {
            /* process the "callCompletion" operation Request in your application */
            if (trace) {
               printf ("Decode of CallCompletionArgument was successful\n");
               callCompletionArgC.Print ("CallCompletionArgument");
            }
         }
         else {
            printf ("Decode of CallCompletionArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 4:   /* clearCall operation Request */
         {
         ASN1T_ClearCallArgument clearCallArg;
         ASN1C_ClearCallArgument clearCallArgC(decodeBuffer2, clearCallArg);
         stat = clearCallArgC.Decode();
         if (stat == 0) {
            /* process the "clearCall" operation Request in your application */
            if (trace) {
               printf ("Decode of ClearCallArgument was successful\n");
               clearCallArgC.Print ("ClearCallArgument");
            }
         }
         else {
            printf ("Decode of ClearCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 5:   /* clearConnection operation Request */
         {
         ASN1T_ClearConnectionArgument clearConnectionArg;
         ASN1C_ClearConnectionArgument clearConnectionArgC(decodeBuffer2, clearConnectionArg);
         stat = clearConnectionArgC.Decode();
         if (stat == 0) {
            /* process the "clearConnection" operation Request in your application */
            if (trace) {
               printf ("Decode of ClearConnectionArgument was successful\n");
               clearConnectionArgC.Print ("ClearConnectionArgument");
            }
         }
         else {
            printf ("Decode of ClearConnectionArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 6:   /* conferenceCall operation Request */
         {
         ASN1T_ConferenceCallArgument conferenceCallArg;
         ASN1C_ConferenceCallArgument conferenceCallArgC(decodeBuffer2, conferenceCallArg);
         stat = conferenceCallArgC.Decode();
         if (stat == 0) {
            /* process the "conferenceCall" operation Request in your application */
            if (trace) {
               printf ("Decode of ConferenceCallArgument was successful\n");
               conferenceCallArgC.Print ("ConferenceCallArgument");
            }
         }
         else {
            printf ("Decode of ConferenceCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 7:   /* consultationCall operation Request */
         {
         ASN1T_ConsultationCallArgument consultationCallArg;
         ASN1C_ConsultationCallArgument consultationCallArgC(decodeBuffer2, consultationCallArg);
         stat = consultationCallArgC.Decode();
         if (stat == 0) {
            /* process the "consultationCall" operation Request in your application */
            if (trace) {
               printf ("Decode of ConsultationCallArgument was successful\n");
               consultationCallArgC.Print ("ConsultationCallArgument");
            }
         }
         else {
            printf ("Decode of ConsultationCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 8:   /* divertCall operation Request */
         {
         ASN1T_DivertCallArgument divertCallArg;
         ASN1C_DivertCallArgument divertCallArgC(decodeBuffer2, divertCallArg);
         stat = divertCallArgC.Decode();
         if (stat == 0) {
            /* process the "divertCall" operation Request in your application */
            if (trace) {
               printf ("Decode of DivertCallArgument was successful\n");
               divertCallArgC.Print ("DivertCallArgument");
            }
         }
         else {
            printf ("Decode of DivertCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 9:   /* holdCall operation Request */
         {
         ASN1T_HoldCallArgument holdCallArg;
         ASN1C_HoldCallArgument holdCallArgC(decodeBuffer2, holdCallArg);
         stat = holdCallArgC.Decode();
         if (stat == 0) {
            /* process the "holdCall" operation Request in your application */
            if (trace) {
               printf ("Decode of HoldCallArgument was successful\n");
               holdCallArgC.Print ("HoldCallArgument");
            }
         }
         else {
            printf ("Decode of HoldCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 10:   /* makeCall operation Request */
         {
         ASN1T_MakeCallArgument makeCallArg;
         ASN1C_MakeCallArgument makeCallArgC(decodeBuffer2, makeCallArg);
         stat = makeCallArgC.Decode();
         if (stat == 0) {
            /* process the "makeCall" operation Request in your application */
            if (trace) {
               printf ("Decode of MakeCallArgument was successful\n");
               makeCallArgC.Print ("MakeCallArgument");
            }
         }
         else {
            printf ("Decode of MakeCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 11:   /* makePredictiveCall operation Request */
         {
         ASN1T_MakePredictiveCallArgument makePredictiveCallArg;
         ASN1C_MakePredictiveCallArgument makePredictiveCallArgC(decodeBuffer2, makePredictiveCallArg);
         stat = makePredictiveCallArgC.Decode();
         if (stat == 0) {
            /* process the "makePredictiveCall" operation Request in your application */
            if (trace) {
               printf ("Decode of MakePredictiveCallArgument was successful\n");
               makePredictiveCallArgC.Print ("MakePredictiveCallArgument");
            }
         }
         else {
            printf ("Decode of MakePredictiveCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 18:   /* parkCall operation Request */
         {
         ASN1T_ParkCallArgument parkCallArg;
         ASN1C_ParkCallArgument parkCallArgC(decodeBuffer2, parkCallArg);
         stat = parkCallArgC.Decode();
         if (stat == 0) {
            /* process the "parkCall" operation Request in your application */
            if (trace) {
               printf ("Decode of ParkCallArgument was successful\n");
               parkCallArgC.Print ("ParkCallArgument");
            }
         }
         else {
            printf ("Decode of ParkCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 12:   /* queryDevice operation Request */
         {
         ASN1T_QueryDeviceArgument queryDeviceArg;
         ASN1C_QueryDeviceArgument queryDeviceArgC(decodeBuffer2, queryDeviceArg);
         stat = queryDeviceArgC.Decode();
         if (stat == 0) {
            /* process the "queryDevice" operation Request in your application */
            if (trace) {
               printf ("Decode of QueryDeviceArgument was successful\n");
               queryDeviceArgC.Print ("QueryDeviceArgument");
            }
         }
         else {
            printf ("Decode of QueryDeviceArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 13:   /* reconnectCall operation Request */
         {
         ASN1T_ReconnectCallArgument reconnectCallArg;
         ASN1C_ReconnectCallArgument reconnectCallArgC(decodeBuffer2, reconnectCallArg);
         stat = reconnectCallArgC.Decode();
         if (stat == 0) {
            /* process the "reconnectCall" operation Request in your application */
            if (trace) {
               printf ("Decode of ReconnectCallArgument was successful\n");
               reconnectCallArgC.Print ("ReconnectCallArgument");
            }
         }
         else {
            printf ("Decode of ReconnectCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 14:   /* retrieveCall operation Request */
         {
         ASN1T_RetrieveCallArgument retrieveCallArg;
         ASN1C_RetrieveCallArgument retrieveCallArgC(decodeBuffer2, retrieveCallArg);
         stat = retrieveCallArgC.Decode();
         if (stat == 0) {
            /* process the "retrieveCall" operation Request in your application */
            if (trace) {
               printf ("Decode of RetrieveCallArgument was successful\n");
               retrieveCallArgC.Print ("RetrieveCallArgument");
            }
         }
         else {
            printf ("Decode of RetrieveCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 19:   /* sendDTMFTones operation Request */
         {
         ASN1T_SendDTMFTonesArgument sendDTMFTonesArg;
         ASN1C_SendDTMFTonesArgument sendDTMFTonesArgC(decodeBuffer2, sendDTMFTonesArg);
         stat = sendDTMFTonesArgC.Decode();
         if (stat == 0) {
            /* process the "sendDTMFTones" operation Request in your application */
            if (trace) {
               printf ("Decode of SendDTMFTonesArgument was successful\n");
               sendDTMFTonesArgC.Print ("SendDTMFTonesArgument");
            }
         }
         else {
            printf ("Decode of SendDTMFTonesArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 15:   /* setFeature operation Request */
         {
         ASN1T_SetFeatureArgument setFeatureArg;
         ASN1C_SetFeatureArgument setFeatureArgC(decodeBuffer2, setFeatureArg);
         stat = setFeatureArgC.Decode();
         if (stat == 0) {
            /* process the "setFeature" operation Request in your application */
            if (trace) {
               printf ("Decode of SetFeatureArgument was successful\n");
               setFeatureArgC.Print ("SetFeatureArgument");
            }
         }
         else {
            printf ("Decode of SetFeatureArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 20:   /* singleStepConf operation Request */
         {
         ASN1T_SingleStepConfArgument singleStepConfArg;
         ASN1C_SingleStepConfArgument singleStepConfArgC(decodeBuffer2, singleStepConfArg);
         stat = singleStepConfArgC.Decode();
         if (stat == 0) {
            /* process the "singleStepConf" operation Request in your application */
            if (trace) {
               printf ("Decode of SingleStepConfArgument was successful\n");
               singleStepConfArgC.Print ("SingleStepConfArgument");
            }
         }
         else {
            printf ("Decode of SingleStepConfArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 50:   /* singleStepTrans operation Request */
         {
         ASN1T_SingleStepTransArgument singleStepTransArg;
         ASN1C_SingleStepTransArgument singleStepTransArgC(decodeBuffer2, singleStepTransArg);
         stat = singleStepTransArgC.Decode();
         if (stat == 0) {
            /* process the "singleStepTrans" operation Request in your application */
            if (trace) {
               printf ("Decode of SingleStepTransArgument was successful\n");
               singleStepTransArgC.Print ("SingleStepTransArgument");
            }
         }
         else {
            printf ("Decode of SingleStepTransArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 16:   /* transferCall operation Request */
         {
         ASN1T_TransferCallArgument transferCallArg;
         ASN1C_TransferCallArgument transferCallArgC(decodeBuffer2, transferCallArg);
         stat = transferCallArgC.Decode();
         if (stat == 0) {
            /* process the "transferCall" operation Request in your application */
            if (trace) {
               printf ("Decode of TransferCallArgument was successful\n");
               transferCallArgC.Print ("TransferCallArgument");
            }
         }
         else {
            printf ("Decode of TransferCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 21:   /* cSTAEventReport operation Request */
         {
         ASN1T_CSTAEventReportArgument cSTAEventReportArg;
         ASN1C_CSTAEventReportArgument cSTAEventReportArgC(decodeBuffer2, cSTAEventReportArg);
         stat = cSTAEventReportArgC.Decode();
         if (stat == 0) {
            if (trace) {
               printf ("Decode of CSTAEventReportArgument was successful\n");
               cSTAEventReportArgC.Print ("CSTAEventReportArgument");
            }
            /* process the "cSTAEventReport" operation Request in your application */
            processEvent(cSTAEventReportArg, trace);
         }
         else {
            printf ("Decode of CSTAEventReportArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 31:   /* routeRequest operation Request */
         {
         ASN1T_RouteRequestArgument routeRequestArg;
         ASN1C_RouteRequestArgument routeRequestArgC(decodeBuffer2, routeRequestArg);
         stat = routeRequestArgC.Decode();
         if (stat == 0) {
            /* process the "routeRequest" operation Request in your application */
            if (trace) {
               printf ("Decode of RouteRequestArgument was successful\n");
               routeRequestArgC.Print ("RouteRequestArgument");
            }
         }
         else {
            printf ("Decode of RouteRequestArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 32:   /* reRouteRequest operation Request */
         {
         ASN1T_ReRouteRequestArgument reRouteRequestArg;
         ASN1C_ReRouteRequestArgument reRouteRequestArgC(decodeBuffer2, reRouteRequestArg);
         stat = reRouteRequestArgC.Decode();
         if (stat == 0) {
            /* process the "reRouteRequest" operation Request in your application */
            if (trace) {
               printf ("Decode of ReRouteRequestArgument was successful\n");
               reRouteRequestArgC.Print ("ReRouteRequestArgument");
            }
         }
         else {
            printf ("Decode of ReRouteRequestArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 33:   /* routeSelectRequest operation Request */
         {
         ASN1T_RouteSelectRequestArgument routeSelectRequestArg;
         ASN1C_RouteSelectRequestArgument routeSelectRequestArgC(decodeBuffer2, routeSelectRequestArg);
         stat = routeSelectRequestArgC.Decode();
         if (stat == 0) {
            /* process the "routeSelectRequest" operation Request in your application */
            if (trace) {
               printf ("Decode of RouteSelectRequestArgument was successful\n");
               routeSelectRequestArgC.Print ("RouteSelectRequestArgument");
            }
         }
         else {
            printf ("Decode of RouteSelectRequestArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 34:   /* routeUsedRequest operation Request */
         {
         ASN1T_RouteUsedRequestArgument routeUsedRequestArg;
         ASN1C_RouteUsedRequestArgument routeUsedRequestArgC(decodeBuffer2, routeUsedRequestArg);
         stat = routeUsedRequestArgC.Decode();
         if (stat == 0) {
            /* process the "routeUsedRequest" operation Request in your application */
            if (trace) {
               printf ("Decode of RouteUsedRequestArgument was successful\n");
               routeUsedRequestArgC.Print ("RouteUsedRequestArgument");
            }
         }
         else {
            printf ("Decode of RouteUsedRequestArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 35:   /* routeEndRequest operation Request */
         {
         ASN1T_RouteEndRequestArgument routeEndRequestArg;
         ASN1C_RouteEndRequestArgument routeEndRequestArgC(decodeBuffer2, routeEndRequestArg);
         stat = routeEndRequestArgC.Decode();
         if (stat == 0) {
            /* process the "routeEndRequest" operation Request in your application */
            if (trace) {
               printf ("Decode of RouteEndRequestArgument was successful\n");
               routeEndRequestArgC.Print ("RouteEndRequestArgument");
            }
         }
         else {
            printf ("Decode of RouteEndRequestArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 51:   /* escapeService operation Request */
         {
         ASN1T_EscapeServiceArgument escapeServiceArg;
         ASN1C_EscapeServiceArgument escapeServiceArgC(decodeBuffer2, escapeServiceArg);
         stat = escapeServiceArgC.Decode();
         if (stat == 0) {
            /* process the "escapeService" operation Request in your application */
            if (trace) {
               printf ("Decode of EscapeServiceArgument was successful\n");
               escapeServiceArgC.Print ("EscapeServiceArgument");
            }
         }
         else {
            printf ("Decode of EscapeServiceArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 52:   /* systemStatus operation Request */
         {
         ASN1T_SystemStatusArgument systemStatusArg;
         ASN1C_SystemStatusArgument systemStatusArgC(decodeBuffer2, systemStatusArg);
         stat = systemStatusArgC.Decode();
         if (stat == 0) {
            processSystemStatusReq(systemStatusArg);
            if (trace) {
               printf ("Decode of SystemStatusArgument was successful\n");
               systemStatusArgC.Print ("SystemStatusArgument");
            }
         }
         else {
            printf ("Decode of SystemStatusArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 71:   /* monitorStart operation Request */
         {
         ASN1T_MonitorStartArgument monitorStartArg;
         ASN1C_MonitorStartArgument monitorStartArgC(decodeBuffer2, monitorStartArg);
         stat = monitorStartArgC.Decode();
         if (stat == 0) {
            /* process the "monitorStart" operation Request in your application */
            if (trace) {
               printf ("Decode of MonitorStartArgument was successful\n");
               monitorStartArgC.Print ("MonitorStartArgument");
            }
         }
         else {
            printf ("Decode of MonitorStartArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 72:   /* changeMonitorFilter operation Request */
         {
         ASN1T_ChangeMonitorFilterArgument changeMonitorFilterArg;
         ASN1C_ChangeMonitorFilterArgument changeMonitorFilterArgC(decodeBuffer2, changeMonitorFilterArg);
         stat = changeMonitorFilterArgC.Decode();
         if (stat == 0) {
            /* process the "changeMonitorFilter" operation Request in your application */
            if (trace) {
               printf ("Decode of ChangeMonitorFilterArgument was successful\n");
               changeMonitorFilterArgC.Print ("ChangeMonitorFilterArgument");
            }
         }
         else {
            printf ("Decode of ChangeMonitorFilterArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 73:   /* monitorStop operation Request */
         {
         ASN1T_MonitorStopArgument monitorStopArg;
         ASN1C_MonitorStopArgument monitorStopArgC(decodeBuffer2, monitorStopArg);
         stat = monitorStopArgC.Decode();
         if (stat == 0) {
            /* process the "monitorStop" operation Request in your application */
            if (trace) {
               printf ("Decode of MonitorStopArgument was successful\n");
               monitorStopArgC.Print ("MonitorStopArgument");
            }
         }
         else {
            printf ("Decode of MonitorStopArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 74:   /* snapshotDevice operation Request */
         {
         ASN1T_SnapshotDeviceArgument snapshotDeviceArg;
         ASN1C_SnapshotDeviceArgument snapshotDeviceArgC(decodeBuffer2, snapshotDeviceArg);
         stat = snapshotDeviceArgC.Decode();
         if (stat == 0) {
            /* process the "snapshotDevice" operation Request in your application */
            if (trace) {
               printf ("Decode of SnapshotDeviceArgument was successful\n");
               snapshotDeviceArgC.Print ("SnapshotDeviceArgument");
            }
         }
         else {
            printf ("Decode of SnapshotDeviceArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 75:   /* snapshotCall operation Request */
         {
         ASN1T_SnapshotCallArgument snapshotCallArg;
         ASN1C_SnapshotCallArgument snapshotCallArgC(decodeBuffer2, snapshotCallArg);
         stat = snapshotCallArgC.Decode();
         if (stat == 0) {
            /* process the "snapshotCall" operation Request in your application */
            if (trace) {
               printf ("Decode of SnapshotCallArgument was successful\n");
               snapshotCallArgC.Print ("SnapshotCallArgument");
            }
         }
         else {
            printf ("Decode of SnapshotCallArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 110:   /* startDataPath operation Request */
         {
         ASN1T_StartDataPathArgument startDataPathArg;
         ASN1C_StartDataPathArgument startDataPathArgC(decodeBuffer2, startDataPathArg);
         stat = startDataPathArgC.Decode();
         if (stat == 0) {
            /* process the "startDataPath" operation Request in your application */
            if (trace) {
               printf ("Decode of StartDataPathArgument was successful\n");
               startDataPathArgC.Print ("StartDataPathArgument");
            }
         }
         else {
            printf ("Decode of StartDataPathArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 111:   /* stopDataPath operation Request */
         {
         ASN1T_StopDataPathArgument stopDataPathArg;
         ASN1C_StopDataPathArgument stopDataPathArgC(decodeBuffer2, stopDataPathArg);
         stat = stopDataPathArgC.Decode();
         if (stat == 0) {
            /* process the "stopDataPath" operation Request in your application */
            if (trace) {
               printf ("Decode of StopDataPathArgument was successful\n");
               stopDataPathArgC.Print ("StopDataPathArgument");
            }
         }
         else {
            printf ("Decode of StopDataPathArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 112:   /* sendData operation Request */
         {
         ASN1T_SendDataArgument sendDataArg;
         ASN1C_SendDataArgument sendDataArgC(decodeBuffer2, sendDataArg);
         stat = sendDataArgC.Decode();
         if (stat == 0) {
            /* process the "sendData" operation Request in your application */
            if (trace) {
               printf ("Decode of SendDataArgument was successful\n");
               sendDataArgC.Print ("SendDataArgument");
            }
         }
         else {
            printf ("Decode of SendDataArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 113:   /* sendMulticastData operation Request */
         {
         ASN1T_SendMulticastDataArgument sendMulticastDataArg;
         ASN1C_SendMulticastDataArgument sendMulticastDataArgC(decodeBuffer2, sendMulticastDataArg);
         stat = sendMulticastDataArgC.Decode();
         if (stat == 0) {
            /* process the "sendMulticastData" operation Request in your application */
            if (trace) {
               printf ("Decode of SendMulticastDataArgument was successful\n");
               sendMulticastDataArgC.Print ("SendMulticastDataArgument");
            }
         }
         else {
            printf ("Decode of SendMulticastDataArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 114:   /* sendBroadcastData operation Request */
         {
         ASN1T_SendBroadcastDataArgument sendBroadcastDataArg;
         ASN1C_SendBroadcastDataArgument sendBroadcastDataArgC(decodeBuffer2, sendBroadcastDataArg);
         stat = sendBroadcastDataArgC.Decode();
         if (stat == 0) {
            /* process the "sendBroadcastData" operation Request in your application */
            if (trace) {
               printf ("Decode of SendBroadcastDataArgument was successful\n");
               sendBroadcastDataArgC.Print ("SendBroadcastDataArgument");
            }
         }
         else {
            printf ("Decode of SendBroadcastDataArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 115:   /* suspendDataPath operation Request */
         {
         ASN1T_SuspendDataPathArgument suspendDataPathArg;
         ASN1C_SuspendDataPathArgument suspendDataPathArgC(decodeBuffer2, suspendDataPathArg);
         stat = suspendDataPathArgC.Decode();
         if (stat == 0) {
            /* process the "suspendDataPath" operation Request in your application */
            if (trace) {
               printf ("Decode of SuspendDataPathArgument was successful\n");
               suspendDataPathArgC.Print ("SuspendDataPathArgument");
            }
         }
         else {
            printf ("Decode of SuspendDataPathArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 116:   /* dataPathSuspended operation Request */
         {
         ASN1T_DataPathSuspendedArgument dataPathSuspendedArg;
         ASN1C_DataPathSuspendedArgument dataPathSuspendedArgC(decodeBuffer2, dataPathSuspendedArg);
         stat = dataPathSuspendedArgC.Decode();
         if (stat == 0) {
            /* process the "dataPathSuspended" operation Request in your application */
            if (trace) {
               printf ("Decode of DataPathSuspendedArgument was successful\n");
               dataPathSuspendedArgC.Print ("DataPathSuspendedArgument");
            }
         }
         else {
            printf ("Decode of DataPathSuspendedArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 117:   /* resumeDataPath operation Request */
         {
         ASN1T_ResumeDataPathArgument resumeDataPathArg;
         ASN1C_ResumeDataPathArgument resumeDataPathArgC(decodeBuffer2, resumeDataPathArg);
         stat = resumeDataPathArgC.Decode();
         if (stat == 0) {
            /* process the "resumeDataPath" operation Request in your application */
            if (trace) {
               printf ("Decode of ResumeDataPathArgument was successful\n");
               resumeDataPathArgC.Print ("ResumeDataPathArgument");
            }
         }
         else {
            printf ("Decode of ResumeDataPathArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 118:   /* dataPathResumed operation Request */
         {
         ASN1T_DataPathResumedArgument dataPathResumedArg;
         ASN1C_DataPathResumedArgument dataPathResumedArgC(decodeBuffer2, dataPathResumedArg);
         stat = dataPathResumedArgC.Decode();
         if (stat == 0) {
            /* process the "dataPathResumed" operation Request in your application */
            if (trace) {
               printf ("Decode of DataPathResumedArgument was successful\n");
               dataPathResumedArgC.Print ("DataPathResumedArgument");
            }
         }
         else {
            printf ("Decode of DataPathResumedArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 119:   /* fastData operation Request */
         {
         ASN1T_CSTA_fast_data_FastDataArgument fastDataArg;
         ASN1C_CSTA_fast_data_FastDataArgument fastDataArgC(decodeBuffer2, fastDataArg);
         stat = fastDataArgC.Decode();
         if (stat == 0) {
            /* process the "fastData" operation Request in your application */
            if (trace) {
               printf ("Decode of FastDataArgument was successful\n");
               fastDataArgC.Print ("FastDataArgument");
            }
         }
         else {
            printf ("Decode of FastDataArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 500:   /* concatenateMessage operation Request */
         {
         ASN1T_ConcatenateMessageArgument concatenateMessageArg;
         ASN1C_ConcatenateMessageArgument concatenateMessageArgC(decodeBuffer2, concatenateMessageArg);
         stat = concatenateMessageArgC.Decode();
         if (stat == 0) {
            /* process the "concatenateMessage" operation Request in your application */
            if (trace) {
               printf ("Decode of ConcatenateMessageArgument was successful\n");
               concatenateMessageArgC.Print ("ConcatenateMessageArgument");
            }
         }
         else {
            printf ("Decode of ConcatenateMessageArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 501:   /* deleteMessage operation Request */
         {
         ASN1T_DeleteMessageArgument deleteMessageArg;
         ASN1C_DeleteMessageArgument deleteMessageArgC(decodeBuffer2, deleteMessageArg);
         stat = deleteMessageArgC.Decode();
         if (stat == 0) {
            /* process the "deleteMessage" operation Request in your application */
            if (trace) {
               printf ("Decode of DeleteMessageArgument was successful\n");
               deleteMessageArgC.Print ("DeleteMessageArgument");
            }
         }
         else {
            printf ("Decode of DeleteMessageArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 502:   /* playMessage operation Request */
         {
         ASN1T_PlayMessageArgument playMessageArg;
         ASN1C_PlayMessageArgument playMessageArgC(decodeBuffer2, playMessageArg);
         stat = playMessageArgC.Decode();
         if (stat == 0) {
            /* process the "playMessage" operation Request in your application */
            if (trace) {
               printf ("Decode of PlayMessageArgument was successful\n");
               playMessageArgC.Print ("PlayMessageArgument");
            }
         }
         else {
            printf ("Decode of PlayMessageArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 503:   /* queryVoiceAttribute operation Request */
         {
         ASN1T_QueryVoiceAttributeArgument queryVoiceAttributeArg;
         ASN1C_QueryVoiceAttributeArgument queryVoiceAttributeArgC(decodeBuffer2, queryVoiceAttributeArg);
         stat = queryVoiceAttributeArgC.Decode();
         if (stat == 0) {
            /* process the "queryVoiceAttribute" operation Request in your application */
            if (trace) {
               printf ("Decode of QueryVoiceAttributeArgument was successful\n");
               queryVoiceAttributeArgC.Print ("QueryVoiceAttributeArgument");
            }
         }
         else {
            printf ("Decode of QueryVoiceAttributeArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 511:   /* recordMessage operation Request */
         {
         ASN1T_RecordMessageArgument recordMessageArg;
         ASN1C_RecordMessageArgument recordMessageArgC(decodeBuffer2, recordMessageArg);
         stat = recordMessageArgC.Decode();
         if (stat == 0) {
            /* process the "recordMessage" operation Request in your application */
            if (trace) {
               printf ("Decode of RecordMessageArgument was successful\n");
               recordMessageArgC.Print ("RecordMessageArgument");
            }
         }
         else {
            printf ("Decode of RecordMessageArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 504:   /* reposition operation Request */
         {
         ASN1T_RepositionArgument repositionArg;
         ASN1C_RepositionArgument repositionArgC(decodeBuffer2, repositionArg);
         stat = repositionArgC.Decode();
         if (stat == 0) {
            /* process the "reposition" operation Request in your application */
            if (trace) {
               printf ("Decode of RepositionArgument was successful\n");
               repositionArgC.Print ("RepositionArgument");
            }
         }
         else {
            printf ("Decode of RepositionArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 505:   /* resume operation Request */
         {
         ASN1T_ResumeArgument resumeArg;
         ASN1C_ResumeArgument resumeArgC(decodeBuffer2, resumeArg);
         stat = resumeArgC.Decode();
         if (stat == 0) {
            /* process the "resume" operation Request in your application */
            if (trace) {
               printf ("Decode of ResumeArgument was successful\n");
               resumeArgC.Print ("ResumeArgument");
            }
         }
         else {
            printf ("Decode of ResumeArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 506:   /* review operation Request */
         {
         ASN1T_ReviewArgument reviewArg;
         ASN1C_ReviewArgument reviewArgC(decodeBuffer2, reviewArg);
         stat = reviewArgC.Decode();
         if (stat == 0) {
            /* process the "review" operation Request in your application */
            if (trace) {
               printf ("Decode of ReviewArgument was successful\n");
               reviewArgC.Print ("ReviewArgument");
            }
         }
         else {
            printf ("Decode of ReviewArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 507:   /* setVoiceAttribute operation Request */
         {
         ASN1T_SetVoiceAttributeArgument setVoiceAttributeArg;
         ASN1C_SetVoiceAttributeArgument setVoiceAttributeArgC(decodeBuffer2, setVoiceAttributeArg);
         stat = setVoiceAttributeArgC.Decode();
         if (stat == 0) {
            /* process the "setVoiceAttribute" operation Request in your application */
            if (trace) {
               printf ("Decode of SetVoiceAttributeArgument was successful\n");
               setVoiceAttributeArgC.Print ("SetVoiceAttributeArgument");
            }
         }
         else {
            printf ("Decode of SetVoiceAttributeArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 508:   /* stop operation Request */
         {
         ASN1T_StopArgument stopArg;
         ASN1C_StopArgument stopArgC(decodeBuffer2, stopArg);
         stat = stopArgC.Decode();
         if (stat == 0) {
            /* process the "stop" operation Request in your application */
            if (trace) {
               printf ("Decode of StopArgument was successful\n");
               stopArgC.Print ("StopArgument");
            }
         }
         else {
            printf ("Decode of StopArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 509:   /* suspend operation Request */
         {
         ASN1T_SuspendArgument suspendArg;
         ASN1C_SuspendArgument suspendArgC(decodeBuffer2, suspendArg);
         stat = suspendArgC.Decode();
         if (stat == 0) {
            /* process the "suspend" operation Request in your application */
            if (trace) {
               printf ("Decode of SuspendArgument was successful\n");
               suspendArgC.Print ("SuspendArgument");
            }
         }
         else {
            printf ("Decode of SuspendArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 510:   /* synthesizeMessage operation Request */
         {
         ASN1T_SynthesizeMessageArgument synthesizeMessageArg;
         ASN1C_SynthesizeMessageArgument synthesizeMessageArgC(decodeBuffer2, synthesizeMessageArg);
         stat = synthesizeMessageArgC.Decode();
         if (stat == 0) {
            /* process the "synthesizeMessage" operation Request in your application */
            if (trace) {
               printf ("Decode of SynthesizeMessageArgument was successful\n");
               synthesizeMessageArgC.Print ("SynthesizeMessageArgument");
            }
         }
         else {
            printf ("Decode of SynthesizeMessageArgument failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;
      default:
         printf ("error: invalid operation code, not defined in csta phase 3\n");
         break;
   }
}

void Client::processEvent(ASN1T_CSTAEventReportArgument& eventReportArg, OSBOOL trace)
{
   //Process the event from EventReportArgument type

}

void Client::processReturnResult(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /*
   ASN1T_ReturnResult.result.result(opentype) needs to be decoded
   using ASN1T_ReturnResult.result.opcode value as defined in Table 1.
   */

   if(pdu.u.returnResult->result.opcode.t != T_Code_local) {
      printf ("ROSE Operation's ReturnResult.result.result type is not known\n");
      return;
   }

   ASN1BERDecodeBuffer decodeBuffer2 (pdu.u.returnResult->result.result.data, 
      pdu.u.returnResult->result.result.numocts);
   int stat;

   switch(pdu.u.returnResult->result.opcode.u.local) {
      case 1 :  /* alternateCall operation Response*/
         {
         ASN1T_AlternateCallResult alternateCallRes;
         ASN1C_AlternateCallResult alternateCallResC(decodeBuffer2, alternateCallRes);
         stat = alternateCallResC.Decode();
         if (stat == 0) {
            /* process the "alternateCall" operation response in your application */
            if (trace) {
               printf ("Decode of AlternateCallResult was successful\n");
               alternateCallResC.Print ("AlternateCallResult");
            }
         }
         else {
            printf ("Decode of AlternateCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 2 :  /* answerCall operation Response*/
         {
         ASN1T_AnswerCallResult answerCallRes;
         ASN1C_AnswerCallResult answerCallResC(decodeBuffer2, answerCallRes);
         stat = answerCallResC.Decode();
         if (stat == 0) {
            processAnswerCallRes(answerCallRes);
            if (trace) {
               printf ("Decode of AnswerCallResult was successful\n");
               answerCallResC.Print ("AnswerCallResult");
            }
         }
         else {
            printf ("Decode of AnswerCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 17 :  /* associateData operation Response*/
         {
         ASN1T_CSTA_associate_data_AssociateDataResult associateDataRes;
         ASN1C_CSTA_associate_data_AssociateDataResult associateDataResC(decodeBuffer2, associateDataRes);
         stat = associateDataResC.Decode();
         if (stat == 0) {
            /* process the "associateData" operation response in your application */
            if (trace) {
               printf ("Decode of AssociateDataResult was successful\n");
               associateDataResC.Print ("AssociateDataResult");
            }
         }
         else {
            printf ("Decode of AssociateDataResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 3 :  /* callCompletion operation Response*/
         {
         ASN1T_CallCompletionResult callCompletionRes;
         ASN1C_CallCompletionResult callCompletionResC(decodeBuffer2, callCompletionRes);
         stat = callCompletionResC.Decode();
         if (stat == 0) {
            /* process the "callCompletion" operation response in your application */
            if (trace) {
               printf ("Decode of CallCompletionResult was successful\n");
               callCompletionResC.Print ("CallCompletionResult");
            }
         }
         else {
            printf ("Decode of CallCompletionResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 4 :  /* clearCall operation Response*/
         {
         ASN1T_ClearCallResult clearCallRes;
         ASN1C_ClearCallResult clearCallResC(decodeBuffer2, clearCallRes);
         stat = clearCallResC.Decode();
         if (stat == 0) {
            /* process the "clearCall" operation response in your application */
            if (trace) {
               printf ("Decode of ClearCallResult was successful\n");
               clearCallResC.Print ("ClearCallResult");
            }
         }
         else {
            printf ("Decode of ClearCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 5 :  /* clearConnection operation Response*/
         {
         ASN1T_ClearConnectionResult clearConnectionRes;
         ASN1C_ClearConnectionResult clearConnectionResC(decodeBuffer2, clearConnectionRes);
         stat = clearConnectionResC.Decode();
         if (stat == 0) {
            /* process the "clearConnection" operation response in your application */
            if (trace) {
               printf ("Decode of ClearConnectionResult was successful\n");
               clearConnectionResC.Print ("ClearConnectionResult");
            }
         }
         else {
            printf ("Decode of ClearConnectionResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 6 :  /* conferenceCall operation Response*/
         {
         ASN1T_ConferenceCallResult conferenceCallRes;
         ASN1C_ConferenceCallResult conferenceCallResC(decodeBuffer2, conferenceCallRes);
         stat = conferenceCallResC.Decode();
         if (stat == 0) {
            /* process the "conferenceCall" operation response in your application */
            if (trace) {
               printf ("Decode of ConferenceCallResult was successful\n");
               conferenceCallResC.Print ("ConferenceCallResult");
            }
         }
         else {
            printf ("Decode of ConferenceCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 7 :  /* consultationCall operation Response*/
         {
         ASN1T_ConsultationCallResult consultationCallRes;
         ASN1C_ConsultationCallResult consultationCallResC(decodeBuffer2, consultationCallRes);
         stat = consultationCallResC.Decode();
         if (stat == 0) {
            /* process the "consultationCall" operation response in your application */
            if (trace) {
               printf ("Decode of ConsultationCallResult was successful\n");
               consultationCallResC.Print ("ConsultationCallResult");
            }
         }
         else {
            printf ("Decode of ConsultationCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 8 :  /* divertCall operation Response*/
         {
         ASN1T_DivertCallResult divertCallRes;
         ASN1C_DivertCallResult divertCallResC(decodeBuffer2, divertCallRes);
         stat = divertCallResC.Decode();
         if (stat == 0) {
            processDivertCallRes(divertCallRes);
            if (trace) {
               printf ("Decode of DivertCallResult was successful\n");
               divertCallResC.Print ("DivertCallResult");
            }
         }
         else {
            printf ("Decode of DivertCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 9 :  /* holdCall operation Response*/
         {
         ASN1T_HoldCallResult holdCallRes;
         ASN1C_HoldCallResult holdCallResC(decodeBuffer2, holdCallRes);
         stat = holdCallResC.Decode();
         if (stat == 0) {
            /* process the "holdCall" operation response in your application */
            if (trace) {
               printf ("Decode of HoldCallResult was successful\n");
               holdCallResC.Print ("HoldCallResult");
            }
         }
         else {
            printf ("Decode of HoldCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 10 :  /* makeCall operation Response*/
         {
         ASN1T_MakeCallResult makeCallRes;
         ASN1C_MakeCallResult makeCallResC(decodeBuffer2, makeCallRes);
         stat = makeCallResC.Decode();
         if (stat == 0) {
            processMakeCallResult(makeCallRes);
            if (trace) {
               printf ("Decode of MakeCallResult was successful\n");
               makeCallResC.Print ("MakeCallResult");
            }
         }
         else {
            printf ("Decode of MakeCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 11 :  /* makePredictiveCall operation Response*/
         {
         ASN1T_MakePredictiveCallResult makePredictiveCallRes;
         ASN1C_MakePredictiveCallResult makePredictiveCallResC(decodeBuffer2, makePredictiveCallRes);
         stat = makePredictiveCallResC.Decode();
         if (stat == 0) {
            /* process the "makePredictiveCall" operation response in your application */
            if (trace) {
               printf ("Decode of MakePredictiveCallResult was successful\n");
               makePredictiveCallResC.Print ("MakePredictiveCallResult");
            }
         }
         else {
            printf ("Decode of MakePredictiveCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 18 :  /* parkCall operation Response*/
         {
         ASN1T_ParkCallResult parkCallRes;
         ASN1C_ParkCallResult parkCallResC(decodeBuffer2, parkCallRes);
         stat = parkCallResC.Decode();
         if (stat == 0) {
            /* process the "parkCall" operation response in your application */
            if (trace) {
               printf ("Decode of ParkCallResult was successful\n");
               parkCallResC.Print ("ParkCallResult");
            }
         }
         else {
            printf ("Decode of ParkCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 12 :  /* queryDevice operation Response*/
         {
         ASN1T_QueryDeviceResult queryDeviceRes;
         ASN1C_QueryDeviceResult queryDeviceResC(decodeBuffer2, queryDeviceRes);
         stat = queryDeviceResC.Decode();
         if (stat == 0) {
            processQueryDeviceRes(queryDeviceRes);
            if (trace) {
               printf ("Decode of QueryDeviceResult was successful\n");
               queryDeviceResC.Print ("QueryDeviceResult");
            }
         }
         else {
            printf ("Decode of QueryDeviceResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 13 :  /* reconnectCall operation Response*/
         {
         ASN1T_ReconnectCallResult reconnectCallRes;
         ASN1C_ReconnectCallResult reconnectCallResC(decodeBuffer2, reconnectCallRes);
         stat = reconnectCallResC.Decode();
         if (stat == 0) {
            /* process the "reconnectCall" operation response in your application */
            if (trace) {
               printf ("Decode of ReconnectCallResult was successful\n");
               reconnectCallResC.Print ("ReconnectCallResult");
            }
         }
         else {
            printf ("Decode of ReconnectCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 14 :  /* retrieveCall operation Response*/
         {
         ASN1T_RetrieveCallResult retrieveCallRes;
         ASN1C_RetrieveCallResult retrieveCallResC(decodeBuffer2, retrieveCallRes);
         stat = retrieveCallResC.Decode();
         if (stat == 0) {
            /* process the "retrieveCall" operation response in your application */
            if (trace) {
               printf ("Decode of RetrieveCallResult was successful\n");
               retrieveCallResC.Print ("RetrieveCallResult");
            }
         }
         else {
            printf ("Decode of RetrieveCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 19 :  /* sendDTMFTones operation Response*/
         {
         ASN1T_SendDTMFTonesResult sendDTMFTonesRes;
         ASN1C_SendDTMFTonesResult sendDTMFTonesResC(decodeBuffer2, sendDTMFTonesRes);
         stat = sendDTMFTonesResC.Decode();
         if (stat == 0) {
            /* process the "sendDTMFTones" operation response in your application */
            if (trace) {
               printf ("Decode of SendDTMFTonesResult was successful\n");
               sendDTMFTonesResC.Print ("SendDTMFTonesResult");
            }
         }
         else {
            printf ("Decode of SendDTMFTonesResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 15 :  /* setFeature operation Response*/
         {
         ASN1T_SetFeatureResult setFeatureRes;
         ASN1C_SetFeatureResult setFeatureResC(decodeBuffer2, setFeatureRes);
         stat = setFeatureResC.Decode();
         if (stat == 0) {
            /* process the "setFeature" operation response in your application */
            if (trace) {
               printf ("Decode of SetFeatureResult was successful\n");
               setFeatureResC.Print ("SetFeatureResult");
            }
         }
         else {
            printf ("Decode of SetFeatureResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 20 :  /* singleStepConf operation Response*/
         {
         ASN1T_SingleStepConfResult singleStepConfRes;
         ASN1C_SingleStepConfResult singleStepConfResC(decodeBuffer2, singleStepConfRes);
         stat = singleStepConfResC.Decode();
         if (stat == 0) {
            /* process the "singleStepConf" operation response in your application */
            if (trace) {
               printf ("Decode of SingleStepConfResult was successful\n");
               singleStepConfResC.Print ("SingleStepConfResult");
            }
         }
         else {
            printf ("Decode of SingleStepConfResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 50 :  /* singleStepTrans operation Response*/
         {
         ASN1T_SingleStepTransResult singleStepTransRes;
         ASN1C_SingleStepTransResult singleStepTransResC(decodeBuffer2, singleStepTransRes);
         stat = singleStepTransResC.Decode();
         if (stat == 0) {
            /* process the "singleStepTrans" operation response in your application */
            if (trace) {
               printf ("Decode of SingleStepTransResult was successful\n");
               singleStepTransResC.Print ("SingleStepTransResult");
            }
         }
         else {
            printf ("Decode of SingleStepTransResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 16 :  /* transferCall operation Response*/
         {
         ASN1T_TransferCallResult transferCallRes;
         ASN1C_TransferCallResult transferCallResC(decodeBuffer2, transferCallRes);
         stat = transferCallResC.Decode();
         if (stat == 0) {
            processTransferCallRes(transferCallRes);
            if (trace) {
               printf ("Decode of TransferCallResult was successful\n");
               transferCallResC.Print ("TransferCallResult");
            }
         }
         else {
            printf ("Decode of TransferCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 51 :  /* escapeService operation Response*/
         {
         ASN1T_EscapeServiceResult escapeServiceRes;
         ASN1C_EscapeServiceResult escapeServiceResC(decodeBuffer2, escapeServiceRes);
         stat = escapeServiceResC.Decode();
         if (stat == 0) {
            /* process the "escapeService" operation response in your application */
            if (trace) {
               printf ("Decode of EscapeServiceResult was successful\n");
               escapeServiceResC.Print ("EscapeServiceResult");
            }
         }
         else {
            printf ("Decode of EscapeServiceResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 52 :  /* systemStatus operation Response*/
         {
         ASN1T_SystemStatusResult systemStatusRes;
         ASN1C_SystemStatusResult systemStatusResC(decodeBuffer2, systemStatusRes);
         stat = systemStatusResC.Decode();
         if (stat == 0) {
            /* process the "systemStatus" operation response in your application */
            if (trace) {
               printf ("Decode of SystemStatusResult was successful\n");
               systemStatusResC.Print ("SystemStatusResult");
            }
         }
         else {
            printf ("Decode of SystemStatusResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 71 :  /* monitorStart operation Response*/
         {
         ASN1T_MonitorStartResult monitorStartRes;
         ASN1C_MonitorStartResult monitorStartResC(decodeBuffer2, monitorStartRes);
         stat = monitorStartResC.Decode();
         if (stat == 0) {
            processMonitorStartRes(monitorStartRes);
            if (trace) {
               printf ("Decode of MonitorStartResult was successful\n");
               monitorStartResC.Print ("MonitorStartResult");
            }
         }
         else {
            printf ("Decode of MonitorStartResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 72 :  /* changeMonitorFilter operation Response*/
         {
         ASN1T_ChangeMonitorFilterResult changeMonitorFilterRes;
         ASN1C_ChangeMonitorFilterResult changeMonitorFilterResC(decodeBuffer2, changeMonitorFilterRes);
         stat = changeMonitorFilterResC.Decode();
         if (stat == 0) {
            /* process the "changeMonitorFilter" operation response in your application */
            if (trace) {
               printf ("Decode of ChangeMonitorFilterResult was successful\n");
               changeMonitorFilterResC.Print ("ChangeMonitorFilterResult");
            }
         }
         else {
            printf ("Decode of ChangeMonitorFilterResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 73 :  /* monitorStop operation Response*/
         {
         ASN1T_MonitorStopResult monitorStopRes;
         ASN1C_MonitorStopResult monitorStopResC(decodeBuffer2, monitorStopRes);
         stat = monitorStopResC.Decode();
         if (stat == 0) {
            processMonitorStopRes(monitorStopRes);
            if (trace) {
               printf ("Decode of MonitorStopResult was successful\n");
               monitorStopResC.Print ("MonitorStopResult");
            }
         }
         else {
            printf ("Decode of MonitorStopResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 74 :  /* snapshotDevice operation Response*/
         {
         ASN1T_SnapshotDeviceResult snapshotDeviceRes;
         ASN1C_SnapshotDeviceResult snapshotDeviceResC(decodeBuffer2, snapshotDeviceRes);
         stat = snapshotDeviceResC.Decode();
         if (stat == 0) {
            /* process the "snapshotDevice" operation response in your application */
            if (trace) {
               printf ("Decode of SnapshotDeviceResult was successful\n");
               snapshotDeviceResC.Print ("SnapshotDeviceResult");
            }
         }
         else {
            printf ("Decode of SnapshotDeviceResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 75 :  /* snapshotCall operation Response*/
         {
         ASN1T_SnapshotCallResult snapshotCallRes;
         ASN1C_SnapshotCallResult snapshotCallResC(decodeBuffer2, snapshotCallRes);
         stat = snapshotCallResC.Decode();
         if (stat == 0) {
            /* process the "snapshotCall" operation response in your application */
            if (trace) {
               printf ("Decode of SnapshotCallResult was successful\n");
               snapshotCallResC.Print ("SnapshotCallResult");
            }
         }
         else {
            printf ("Decode of SnapshotCallResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 110 :  /* startDataPath operation Response*/
         {
         ASN1T_StartDataPathResult startDataPathRes;
         ASN1C_StartDataPathResult startDataPathResC(decodeBuffer2, startDataPathRes);
         stat = startDataPathResC.Decode();
         if (stat == 0) {
            /* process the "startDataPath" operation response in your application */
            if (trace) {
               printf ("Decode of StartDataPathResult was successful\n");
               startDataPathResC.Print ("StartDataPathResult");
            }
         }
         else {
            printf ("Decode of StartDataPathResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 111 :  /* stopDataPath operation Response*/
         {
         ASN1T_StopDataPathResult stopDataPathRes;
         ASN1C_StopDataPathResult stopDataPathResC(decodeBuffer2, stopDataPathRes);
         stat = stopDataPathResC.Decode();
         if (stat == 0) {
            /* process the "stopDataPath" operation response in your application */
            if (trace) {
               printf ("Decode of StopDataPathResult was successful\n");
               stopDataPathResC.Print ("StopDataPathResult");
            }
         }
         else {
            printf ("Decode of StopDataPathResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 112 :  /* sendData operation Response*/
         {
         ASN1T_SendDataResult sendDataRes;
         ASN1C_SendDataResult sendDataResC(decodeBuffer2, sendDataRes);
         stat = sendDataResC.Decode();
         if (stat == 0) {
            /* process the "sendData" operation response in your application */
            if (trace) {
               printf ("Decode of SendDataResult was successful\n");
               sendDataResC.Print ("SendDataResult");
            }
         }
         else {
            printf ("Decode of SendDataResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 113 :  /* sendMulticastData operation Response*/
         {
         ASN1T_SendMulticastDataResult sendMulticastDataRes;
         ASN1C_SendMulticastDataResult sendMulticastDataResC(decodeBuffer2, sendMulticastDataRes);
         stat = sendMulticastDataResC.Decode();
         if (stat == 0) {
            /* process the "sendMulticastData" operation response in your application */
            if (trace) {
               printf ("Decode of SendMulticastDataResult was successful\n");
               sendMulticastDataResC.Print ("SendMulticastDataResult");
            }
         }
         else {
            printf ("Decode of SendMulticastDataResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 114 :  /* sendBroadcastData operation Response*/
         {
         ASN1T_SendBroadcastDataResult sendBroadcastDataRes;
         ASN1C_SendBroadcastDataResult sendBroadcastDataResC(decodeBuffer2, sendBroadcastDataRes);
         stat = sendBroadcastDataResC.Decode();
         if (stat == 0) {
            /* process the "sendBroadcastData" operation response in your application */
            if (trace) {
               printf ("Decode of SendBroadcastDataResult was successful\n");
               sendBroadcastDataResC.Print ("SendBroadcastDataResult");
            }
         }
         else {
            printf ("Decode of SendBroadcastDataResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 115 :  /* suspendDataPath operation Response*/
         {
         ASN1T_SuspendDataPathResult suspendDataPathRes;
         ASN1C_SuspendDataPathResult suspendDataPathResC(decodeBuffer2, suspendDataPathRes);
         stat = suspendDataPathResC.Decode();
         if (stat == 0) {
            /* process the "suspendDataPath" operation response in your application */
            if (trace) {
               printf ("Decode of SuspendDataPathResult was successful\n");
               suspendDataPathResC.Print ("SuspendDataPathResult");
            }
         }
         else {
            printf ("Decode of SuspendDataPathResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 116 :  /* dataPathSuspended operation Response*/
         {
         ASN1T_DataPathSuspendedResult dataPathSuspendedRes;
         ASN1C_DataPathSuspendedResult dataPathSuspendedResC(decodeBuffer2, dataPathSuspendedRes);
         stat = dataPathSuspendedResC.Decode();
         if (stat == 0) {
            /* process the "dataPathSuspended" operation response in your application */
            if (trace) {
               printf ("Decode of DataPathSuspendedResult was successful\n");
               dataPathSuspendedResC.Print ("DataPathSuspendedResult");
            }
         }
         else {
            printf ("Decode of DataPathSuspendedResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 117 :  /* resumeDataPath operation Response*/
         {
         ASN1T_ResumeDataPathResult resumeDataPathRes;
         ASN1C_ResumeDataPathResult resumeDataPathResC(decodeBuffer2, resumeDataPathRes);
         stat = resumeDataPathResC.Decode();
         if (stat == 0) {
            /* process the "resumeDataPath" operation response in your application */
            if (trace) {
               printf ("Decode of ResumeDataPathResult was successful\n");
               resumeDataPathResC.Print ("ResumeDataPathResult");
            }
         }
         else {
            printf ("Decode of ResumeDataPathResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 118 :  /* dataPathResumed operation Response*/
         {
         ASN1T_DataPathResumedResult dataPathResumedRes;
         ASN1C_DataPathResumedResult dataPathResumedResC(decodeBuffer2, dataPathResumedRes);
         stat = dataPathResumedResC.Decode();
         if (stat == 0) {
            /* process the "dataPathResumed" operation response in your application */
            if (trace) {
               printf ("Decode of DataPathResumedResult was successful\n");
               dataPathResumedResC.Print ("DataPathResumedResult");
            }
         }
         else {
            printf ("Decode of DataPathResumedResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 119 :  /* fastData operation Response*/
         {
         ASN1T_CSTA_fast_data_FastDataResult fastDataRes;
         ASN1C_CSTA_fast_data_FastDataResult fastDataResC(decodeBuffer2, fastDataRes);
         stat = fastDataResC.Decode();
         if (stat == 0) {
            /* process the "fastData" operation response in your application */
            if (trace) {
               printf ("Decode of FastDataResult was successful\n");
               fastDataResC.Print ("FastDataResult");
            }
         }
         else {
            printf ("Decode of FastDataResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 500 :  /* concatenateMessage operation Response*/
         {
         ASN1T_ConcatenateMessageResult concatenateMessageRes;
         ASN1C_ConcatenateMessageResult concatenateMessageResC(decodeBuffer2, concatenateMessageRes);
         stat = concatenateMessageResC.Decode();
         if (stat == 0) {
            /* process the "concatenateMessage" operation response in your application */
            if (trace) {
               printf ("Decode of ConcatenateMessageResult was successful\n");
               concatenateMessageResC.Print ("ConcatenateMessageResult");
            }
         }
         else {
            printf ("Decode of ConcatenateMessageResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 501 :  /* deleteMessage operation Response*/
         {
         ASN1T_DeleteMessageResult deleteMessageRes;
         ASN1C_DeleteMessageResult deleteMessageResC(decodeBuffer2, deleteMessageRes);
         stat = deleteMessageResC.Decode();
         if (stat == 0) {
            /* process the "deleteMessage" operation response in your application */
            if (trace) {
               printf ("Decode of DeleteMessageResult was successful\n");
               deleteMessageResC.Print ("DeleteMessageResult");
            }
         }
         else {
            printf ("Decode of DeleteMessageResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 502 :  /* playMessage operation Response*/
         {
         ASN1T_PlayMessageResult playMessageRes;
         ASN1C_PlayMessageResult playMessageResC(decodeBuffer2, playMessageRes);
         stat = playMessageResC.Decode();
         if (stat == 0) {
            /* process the "playMessage" operation response in your application */
            if (trace) {
               printf ("Decode of PlayMessageResult was successful\n");
               playMessageResC.Print ("PlayMessageResult");
            }
         }
         else {
            printf ("Decode of PlayMessageResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 503 :  /* queryVoiceAttribute operation Response*/
         {
         ASN1T_QueryVoiceAttributeResult queryVoiceAttributeRes;
         ASN1C_QueryVoiceAttributeResult queryVoiceAttributeResC(decodeBuffer2, queryVoiceAttributeRes);
         stat = queryVoiceAttributeResC.Decode();
         if (stat == 0) {
            /* process the "queryVoiceAttribute" operation response in your application */
            if (trace) {
               printf ("Decode of QueryVoiceAttributeResult was successful\n");
               queryVoiceAttributeResC.Print ("QueryVoiceAttributeResult");
            }
         }
         else {
            printf ("Decode of QueryVoiceAttributeResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 511 :  /* recordMessage operation Response*/
         {
         ASN1T_RecordMessageResult recordMessageRes;
         ASN1C_RecordMessageResult recordMessageResC(decodeBuffer2, recordMessageRes);
         stat = recordMessageResC.Decode();
         if (stat == 0) {
            /* process the "recordMessage" operation response in your application */
            if (trace) {
               printf ("Decode of RecordMessageResult was successful\n");
               recordMessageResC.Print ("RecordMessageResult");
            }
         }
         else {
            printf ("Decode of RecordMessageResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 504 :  /* reposition operation Response*/
         {
         ASN1T_RepositionResult repositionRes;
         ASN1C_RepositionResult repositionResC(decodeBuffer2, repositionRes);
         stat = repositionResC.Decode();
         if (stat == 0) {
            /* process the "reposition" operation response in your application */
            if (trace) {
               printf ("Decode of RepositionResult was successful\n");
               repositionResC.Print ("RepositionResult");
            }
         }
         else {
            printf ("Decode of RepositionResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 505 :  /* resume operation Response*/
         {
         ASN1T_ResumeResult resumeRes;
         ASN1C_ResumeResult resumeResC(decodeBuffer2, resumeRes);
         stat = resumeResC.Decode();
         if (stat == 0) {
            /* process the "resume" operation response in your application */
            if (trace) {
               printf ("Decode of ResumeResult was successful\n");
               resumeResC.Print ("ResumeResult");
            }
         }
         else {
            printf ("Decode of ResumeResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 506 :  /* review operation Response*/
         {
         ASN1T_ReviewResult reviewRes;
         ASN1C_ReviewResult reviewResC(decodeBuffer2, reviewRes);
         stat = reviewResC.Decode();
         if (stat == 0) {
            /* process the "review" operation response in your application */
            if (trace) {
               printf ("Decode of ReviewResult was successful\n");
               reviewResC.Print ("ReviewResult");
            }
         }
         else {
            printf ("Decode of ReviewResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 507 :  /* setVoiceAttribute operation Response*/
         {
         ASN1T_SetVoiceAttributeResult setVoiceAttributeRes;
         ASN1C_SetVoiceAttributeResult setVoiceAttributeResC(decodeBuffer2, setVoiceAttributeRes);
         stat = setVoiceAttributeResC.Decode();
         if (stat == 0) {
            /* process the "setVoiceAttribute" operation response in your application */
            if (trace) {
               printf ("Decode of SetVoiceAttributeResult was successful\n");
               setVoiceAttributeResC.Print ("SetVoiceAttributeResult");
            }
         }
         else {
            printf ("Decode of SetVoiceAttributeResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 508 :  /* stop operation Response*/
         {
         ASN1T_StopResult stopRes;
         ASN1C_StopResult stopResC(decodeBuffer2, stopRes);
         stat = stopResC.Decode();
         if (stat == 0) {
            /* process the "stop" operation response in your application */
            if (trace) {
               printf ("Decode of StopResult was successful\n");
               stopResC.Print ("StopResult");
            }
         }
         else {
            printf ("Decode of StopResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 509 :  /* suspend operation Response*/
         {
         ASN1T_SuspendResult suspendRes;
         ASN1C_SuspendResult suspendResC(decodeBuffer2, suspendRes);
         stat = suspendResC.Decode();
         if (stat == 0) {
            /* process the "suspend" operation response in your application */
            if (trace) {
               printf ("Decode of SuspendResult was successful\n");
               suspendResC.Print ("SuspendResult");
            }
         }
         else {
            printf ("Decode of SuspendResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      case 510 :  /* synthesizeMessage operation Response*/
         {
         ASN1T_SynthesizeMessageResult synthesizeMessageRes;
         ASN1C_SynthesizeMessageResult synthesizeMessageResC(decodeBuffer2, synthesizeMessageRes);
         stat = synthesizeMessageResC.Decode();
         if (stat == 0) {
            /* process the "synthesizeMessage" operation response in your application */
            if (trace) {
               printf ("Decode of SynthesizeMessageResult was successful\n");
               synthesizeMessageResC.Print ("SynthesizeMessageResult");
            }
         }
         else {
            printf ("Decode of SynthesizeMessageResult failed\n");
            decodeBuffer2.printErrorInfo ();
            return;
         }
      }
         break;

      default: /* error: invalid operation code, not defined in csta phase 3*/
         printf ("error: invalid operation code, not defined in csta phase 3\n");
         break;
   }
}



void Client::processReturnError(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /* 
   return error processing will require to decode the ASN1T_ReturnError.parameter
   opentype in to real error type, only one possible error type can be returned
   in csta 2, which is "UniversalFailure" 
   
   ASN1T_ReturnError.errcode field is used to findout the real type for 
   ASN1T_ReturnError.parameter(opentype). Real type for all the operation
   in Table 1(i.e csta3) is "UniversalFailure"
   */
   int stat;
   ASN1BERDecodeBuffer decodeBuffer2 
      (pdu.u.returnError->parameter.data, pdu.u.returnError->parameter.numocts);
   ASN1T_UniversalFailure uniFailure;
   ASN1C_UniversalFailure uniFailureC (decodeBuffer2, uniFailure);
   stat = uniFailureC.Decode ();
   if (stat == 0) {
      /* process the ReturnError in your application */
      if (trace) {
         printf ("Decode of ReturnError was successful\n");
         uniFailureC.Print ("UniversalFailure");
      }
   }
   else {
      printf ("Decode of ReturnError failed\n");
      decodeBuffer2.printErrorInfo ();
      return ;
   }
}



void Client::processReject(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL trace)
{
   /* process the reject in your application */
}

