/* Test driver to read and decode an encoded ASN.1 message 
 * This file provides sample code for processing the PBX response
 *
 * @Author Avin Patel
 */

#include <stdio.h>
#include <stdlib.h>
#include "client.h"

int Client::receiveCSTAMessage(OSRTSocket& cstaSocket)
{
   do {
      ASN1BERDecodeBuffer decodeBuffer;
      if(receiveMessage(cstaSocket, decodeBuffer) < 0) {
         printf("error: failed to receive message\n");
         return -1;
      }

      ASN1T_CSTA_ROSE_PDU pdu;
      ASN1C_CSTA_ROSE_PDU pduC (decodeBuffer, pdu);

      int stat = pduC.Decode ();

      if (stat == 0) {
         if (trace) {
            printf ("Decode of ROSE PDU was successful\n");
            pduC.Print ("pdu");
         }
      }
      else {
         printf ("Decode of ROSE PDU failed\n");
         decodeBuffer.printErrorInfo ();
         return -1;
      }

      /* Process the ROSE operation */
      processPDU (pdu, trace);
      printf ("***\n");

   } while (eventReceived);

   return 0;
}

void Client::processPDU(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL tracearg)
{
   eventReceived = FALSE;
   switch(pdu.t){
      case T_CSTA_ROSE_PDU_invoke: {
         processInvoke(pdu, tracearg);
         } break;
      
      case T_CSTA_ROSE_PDU_returnResult: {
         processReturnResult(pdu, tracearg);
         } break;

      case T_CSTA_ROSE_PDU_returnError: {
         processReturnError(pdu, tracearg);
         } break;
         
      case T_CSTA_ROSE_PDU_reject: {
         processReject(pdu, tracearg);
         } break;

      default: { } break;
   }
}

void Client::processInvoke(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL tracearg)
{
   /*
   ASN1T_CSTA_ROSE_PDU_invoke.argument(opentype) needs to be decoded/encoded
   using ASN1T_CSTA_ROSE_PDU_invoke.opcode value as defined in Table 1.
   */

   if(pdu.u.invoke->opcode.t != T_Code_local) {
      printf ("ROSE Operation's Invoke.parameter type is not known\n");
      return;
   }
   
   ASN1BERDecodeBuffer decodeBuffer2 
      (pdu.u.invoke->argument.data, pdu.u.invoke->argument.numocts);
   int stat;
   switch(pdu.u.invoke->opcode.u.local) {
      case 1 : {  /* alternateCall operation */
            ASN1T_AlternateCallArgument alternateCallArg;
            ASN1C_AlternateCallArgument alternateCallArgC(decodeBuffer2, alternateCallArg);
            stat = alternateCallArgC.Decode();
            if (stat == 0) {
               /* process the alternateCall operation in your application */
               if (tracearg) {
                  printf ("Decode of AlternateCallArgument was successful\n");
                  alternateCallArgC.Print ("AlternateCallArgument");
               }
            }
            else {
               printf ("Decode of AlternateCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 2 : {  /* answerCall operation */
            ASN1T_AnswerCallArgument answerCallArg;
            ASN1C_AnswerCallArgument answerCallArgC(decodeBuffer2, answerCallArg);
            stat = answerCallArgC.Decode();
            if (stat == 0) {
               /* process the answerCall operation in your application */
               if (tracearg) {
                  printf ("Decode of AnswerCallArgument was successful\n");
                  answerCallArgC.Print ("AnswerCallArgument");
               }
            }
            else {
               printf ("Decode of AnswerCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 4 : {  /* clearCall operation */
            ASN1T_ClearCallArgument clearCallArg;
            ASN1C_ClearCallArgument clearCallArgC(decodeBuffer2, clearCallArg);
            stat = clearCallArgC.Decode();
            if (stat == 0) {
               /* process the clearCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ClearCallArgument was successful\n");
                  clearCallArgC.Print ("ClearCallArgument");
               }
            }
            else {
               printf ("Decode of ClearCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 5 : {  /* clearConnection operation */
            ASN1T_ClearConnectionArgument clearConnectionArg;
            ASN1C_ClearConnectionArgument clearConnectionArgC(decodeBuffer2, clearConnectionArg);
            stat = clearConnectionArgC.Decode();
            if (stat == 0) {
               /* process the clearConnection operation in your application */
               if (tracearg) {
                  printf ("Decode of ClearConnectionArgument was successful\n");
                  clearConnectionArgC.Print ("ClearConnectionArgument");
               }
            }
            else {
               printf ("Decode of ClearConnectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 6 : {  /* conferenceCall operation */
            ASN1T_ConferenceCallArgument conferenceCallArg;
            ASN1C_ConferenceCallArgument conferenceCallArgC(decodeBuffer2, conferenceCallArg);
            stat = conferenceCallArgC.Decode();
            if (stat == 0) {
               /* process the conferenceCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ConferenceCallArgument was successful\n");
                  conferenceCallArgC.Print ("ConferenceCallArgument");
               }
            }
            else {
               printf ("Decode of ConferenceCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 7 : {  /* consultationCall operation */
            ASN1T_ConsultationCallArgument consultationCallArg;
            ASN1C_ConsultationCallArgument consultationCallArgC(decodeBuffer2, consultationCallArg);
            stat = consultationCallArgC.Decode();
            if (stat == 0) {
               /* process the consultationCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ConsultationCallArgument was successful\n");
                  consultationCallArgC.Print ("ConsultationCallArgument");
               }
            }
            else {
               printf ("Decode of ConsultationCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 9 : {  /* holdCall operation */
            ASN1T_HoldCallArgument holdCallArg;
            ASN1C_HoldCallArgument holdCallArgC(decodeBuffer2, holdCallArg);
            stat = holdCallArgC.Decode();
            if (stat == 0) {
               /* process the holdCall operation in your application */
               if (tracearg) {
                  printf ("Decode of HoldCallArgument was successful\n");
                  holdCallArgC.Print ("HoldCallArgument");
               }
            }
            else {
               printf ("Decode of HoldCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 10 : {  /* makeCall operation */
            ASN1T_MakeCallArgument makeCallArg;
            ASN1C_MakeCallArgument makeCallArgC(decodeBuffer2, makeCallArg);
            stat = makeCallArgC.Decode();
            if (stat == 0) {
               /* process the makeCall operation in your application */
               if (tracearg) {
                  printf ("Decode of MakeCallArgument was successful\n");
                  makeCallArgC.Print ("MakeCallArgument");
               }
            }
            else {
               printf ("Decode of MakeCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 11 : {  /* makePredictiveCall operation */
            ASN1T_MakePredictiveCallArgument makePredictiveCallArg;
            ASN1C_MakePredictiveCallArgument makePredictiveCallArgC(decodeBuffer2, makePredictiveCallArg);
            stat = makePredictiveCallArgC.Decode();
            if (stat == 0) {
               /* process the makePredictiveCall operation in your application */
               if (tracearg) {
                  printf ("Decode of MakePredictiveCallArgument was successful\n");
                  makePredictiveCallArgC.Print ("MakePredictiveCallArgument");
               }
            }
            else {
               printf ("Decode of MakePredictiveCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 13 : {  /* reconnectCall operation */
            ASN1T_ReconnectCallArgument reconnectCallArg;
            ASN1C_ReconnectCallArgument reconnectCallArgC(decodeBuffer2, reconnectCallArg);
            stat = reconnectCallArgC.Decode();
            if (stat == 0) {
               /* process the reconnectCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ReconnectCallArgument was successful\n");
                  reconnectCallArgC.Print ("ReconnectCallArgument");
               }
            }
            else {
               printf ("Decode of ReconnectCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 14 : {  /* retrieveCall operation */
            ASN1T_RetrieveCallArgument retrieveCallArg;
            ASN1C_RetrieveCallArgument retrieveCallArgC(decodeBuffer2, retrieveCallArg);
            stat = retrieveCallArgC.Decode();
            if (stat == 0) {
               /* process the retrieveCall operation in your application */
               if (tracearg) {
                  printf ("Decode of RetrieveCallArgument was successful\n");
                  retrieveCallArgC.Print ("RetrieveCallArgument");
               }
            }
            else {
               printf ("Decode of RetrieveCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 16 : {  /* transferCall operation */
            ASN1T_TransferCallArgument transferCallArg;
            ASN1C_TransferCallArgument transferCallArgC(decodeBuffer2, transferCallArg);
            stat = transferCallArgC.Decode();
            if (stat == 0) {
               /* process the transferCall operation in your application */
               if (tracearg) {
                  printf ("Decode of TransferCallArgument was successful\n");
                  transferCallArgC.Print ("TransferCallArgument");
               }
            }
            else {
               printf ("Decode of TransferCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 18 : {  /* parkCall operation */
            ASN1T_ParkCallArgument parkCallArg;
            ASN1C_ParkCallArgument parkCallArgC(decodeBuffer2, parkCallArg);
            stat = parkCallArgC.Decode();
            if (stat == 0) {
               /* process the parkCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ParkCallArgument was successful\n");
                  parkCallArgC.Print ("ParkCallArgument");
               }
            }
            else {
               printf ("Decode of ParkCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 20 : {  /* singleStepConf operation */
            ASN1T_SingleStepConfArgument singleStepConfArg;
            ASN1C_SingleStepConfArgument singleStepConfArgC(decodeBuffer2, singleStepConfArg);
            stat = singleStepConfArgC.Decode();
            if (stat == 0) {
               /* process the singleStepConf operation in your application */
               if (tracearg) {
                  printf ("Decode of SingleStepConfArgument was successful\n");
                  singleStepConfArgC.Print ("SingleStepConfArgument");
               }
            }
            else {
               printf ("Decode of SingleStepConfArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 21 : {  /* cSTAEventReport operation */
            ASN1T_cerd_CSTAEventReportArgument cSTAEventReportArg;
            ASN1C_cerd_CSTAEventReportArgument cSTAEventReportArgC(decodeBuffer2, cSTAEventReportArg);
            stat = cSTAEventReportArgC.Decode();
            if (stat == 0) {
               if (tracearg) {
                  printf ("Decode of CSTAEventReportArgument was successful\n");
                  cSTAEventReportArgC.Print ("CSTAEventReportArgument");
               }
               /* process the "cSTAEventReport" operation Request in your application */
               processEvent(cSTAEventReportArg, trace);
            }
            else {
               printf ("Decode of CSTAEventReportArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
            eventReceived = TRUE;
         } break;

      case 31 : {  /* routeRequest operation */
            ASN1T_RouteRequestArgument routeRequestArg;
            ASN1C_RouteRequestArgument routeRequestArgC(decodeBuffer2, routeRequestArg);
            stat = routeRequestArgC.Decode();
            if (stat == 0) {
               /* process the routeRequest operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRequestArgument was successful\n");
                  routeRequestArgC.Print ("RouteRequestArgument");
               }
            }
            else {
               printf ("Decode of RouteRequestArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 32 : {  /* reRouteRequest operation */
            ASN1T_ReRouteRequestArgument reRouteRequestArg;
            ASN1C_ReRouteRequestArgument reRouteRequestArgC(decodeBuffer2, reRouteRequestArg);
            stat = reRouteRequestArgC.Decode();
            if (stat == 0) {
               /* process the reRouteRequest operation in your application */
               if (tracearg) {
                  printf ("Decode of ReRouteRequestArgument was successful\n");
                  reRouteRequestArgC.Print ("ReRouteRequestArgument");
               }
            }
            else {
               printf ("Decode of ReRouteRequestArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 33 : {  /* routeSelectRequest operation */
            ASN1T_RouteSelectRequestArgument routeSelectRequestArg;
            ASN1C_RouteSelectRequestArgument routeSelectRequestArgC(decodeBuffer2, routeSelectRequestArg);
            stat = routeSelectRequestArgC.Decode();
            if (stat == 0) {
               /* process the routeSelectRequest operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteSelectRequestArgument was successful\n");
                  routeSelectRequestArgC.Print ("RouteSelectRequestArgument");
               }
            }
            else {
               printf ("Decode of RouteSelectRequestArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 34 : {  /* routeUsedRequest operation */
            ASN1T_RouteUsedRequestArgument routeUsedRequestArg;
            ASN1C_RouteUsedRequestArgument routeUsedRequestArgC(decodeBuffer2, routeUsedRequestArg);
            stat = routeUsedRequestArgC.Decode();
            if (stat == 0) {
               /* process the routeUsedRequest operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteUsedRequestArgument was successful\n");
                  routeUsedRequestArgC.Print ("RouteUsedRequestArgument");
               }
            }
            else {
               printf ("Decode of RouteUsedRequestArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 35 : {  /* routeEndRequest operation */
            ASN1T_RouteEndRequestArgument routeEndRequestArg;
            ASN1C_RouteEndRequestArgument routeEndRequestArgC(decodeBuffer2, routeEndRequestArg);
            stat = routeEndRequestArgC.Decode();
            if (stat == 0) {
               /* process the routeEndRequest operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteEndRequestArgument was successful\n");
                  routeEndRequestArgC.Print ("RouteEndRequestArgument");
               }
            }
            else {
               printf ("Decode of RouteEndRequestArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 36 : {  /* routeReject operation */
            ASN1T_RouteRejectArgument routeRejectArg;
            ASN1C_RouteRejectArgument routeRejectArgC(decodeBuffer2, routeRejectArg);
            stat = routeRejectArgC.Decode();
            if (stat == 0) {
               /* process the routeReject operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRejectArgument was successful\n");
                  routeRejectArgC.Print ("RouteRejectArgument");
               }
            }
            else {
               printf ("Decode of RouteRejectArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 50 : {  /* singleStepTrans operation */
            ASN1T_SingleStepTransArgument singleStepTransArg;
            ASN1C_SingleStepTransArgument singleStepTransArgC(decodeBuffer2, singleStepTransArg);
            stat = singleStepTransArgC.Decode();
            if (stat == 0) {
               /* process the singleStepTrans operation in your application */
               if (tracearg) {
                  printf ("Decode of SingleStepTransArgument was successful\n");
                  singleStepTransArgC.Print ("SingleStepTransArgument");
               }
            }
            else {
               printf ("Decode of SingleStepTransArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 51 : {  /* escape operation */
            ASN1T_EscapeArgument escapeArg;
            ASN1C_EscapeArgument escapeArgC(decodeBuffer2, escapeArg);
            stat = escapeArgC.Decode();
            if (stat == 0) {
               /* process the escape operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeArgument was successful\n");
                  escapeArgC.Print ("EscapeArgument");
               }
            }
            else {
               printf ("Decode of EscapeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 71 : {  /* monitorStart operation */
            ASN1T_MonitorStartArgument monitorStartArg;
            ASN1C_MonitorStartArgument monitorStartArgC(decodeBuffer2, monitorStartArg);
            stat = monitorStartArgC.Decode();
            if (stat == 0) {
               /* process the monitorStart operation in your application */
               if (tracearg) {
                  printf ("Decode of MonitorStartArgument was successful\n");
                  monitorStartArgC.Print ("MonitorStartArgument");
               }
            }
            else {
               printf ("Decode of MonitorStartArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 72 : {  /* changeMonitorFilter operation */
            ASN1T_ChangeMonitorFilterArgument changeMonitorFilterArg;
            ASN1C_ChangeMonitorFilterArgument changeMonitorFilterArgC(decodeBuffer2, changeMonitorFilterArg);
            stat = changeMonitorFilterArgC.Decode();
            if (stat == 0) {
               /* process the changeMonitorFilter operation in your application */
               if (tracearg) {
                  printf ("Decode of ChangeMonitorFilterArgument was successful\n");
                  changeMonitorFilterArgC.Print ("ChangeMonitorFilterArgument");
               }
            }
            else {
               printf ("Decode of ChangeMonitorFilterArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 73 : {  /* monitorStop operation */
            ASN1T_MonitorStopArgument monitorStopArg;
            ASN1C_MonitorStopArgument monitorStopArgC(decodeBuffer2, monitorStopArg);
            stat = monitorStopArgC.Decode();
            if (stat == 0) {
               /* process the monitorStop operation in your application */
               if (tracearg) {
                  printf ("Decode of MonitorStopArgument was successful\n");
                  monitorStopArgC.Print ("MonitorStopArgument");
               }
            }
            else {
               printf ("Decode of MonitorStopArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 74 : {  /* snapshotDevice operation */
            ASN1T_SnapshotDeviceArgument snapshotDeviceArg;
            ASN1C_SnapshotDeviceArgument snapshotDeviceArgC(decodeBuffer2, snapshotDeviceArg);
            stat = snapshotDeviceArgC.Decode();
            if (stat == 0) {
               /* process the snapshotDevice operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotDeviceArgument was successful\n");
                  snapshotDeviceArgC.Print ("SnapshotDeviceArgument");
               }
            }
            else {
               printf ("Decode of SnapshotDeviceArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 75 : {  /* snapshotCall operation */
            ASN1T_SnapshotCallArgument snapshotCallArg;
            ASN1C_SnapshotCallArgument snapshotCallArgC(decodeBuffer2, snapshotCallArg);
            stat = snapshotCallArgC.Decode();
            if (stat == 0) {
               /* process the snapshotCall operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotCallArgument was successful\n");
                  snapshotCallArgC.Print ("SnapshotCallArgument");
               }
            }
            else {
               printf ("Decode of SnapshotCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 76 : {  /* snapshotCallData operation */
            ASN1T_SnapshotCallDataArgument snapshotCallDataArg;
            ASN1C_SnapshotCallDataArgument snapshotCallDataArgC(decodeBuffer2, snapshotCallDataArg);
            stat = snapshotCallDataArgC.Decode();
            if (stat == 0) {
               /* process the snapshotCallData operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotCallDataArgument was successful\n");
                  snapshotCallDataArgC.Print ("SnapshotCallDataArgument");
               }
            }
            else {
               printf ("Decode of SnapshotCallDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 77 : {  /* snapshotDeviceData operation */
            ASN1T_SnapshotDeviceDataArgument snapshotDeviceDataArg;
            ASN1C_SnapshotDeviceDataArgument snapshotDeviceDataArgC(decodeBuffer2, snapshotDeviceDataArg);
            stat = snapshotDeviceDataArgC.Decode();
            if (stat == 0) {
               /* process the snapshotDeviceData operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotDeviceDataArgument was successful\n");
                  snapshotDeviceDataArgC.Print ("SnapshotDeviceDataArgument");
               }
            }
            else {
               printf ("Decode of SnapshotDeviceDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 110 : {  /* startDataPath operation */
            ASN1T_StartDataPathArgument startDataPathArg;
            ASN1C_StartDataPathArgument startDataPathArgC(decodeBuffer2, startDataPathArg);
            stat = startDataPathArgC.Decode();
            if (stat == 0) {
               /* process the startDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of StartDataPathArgument was successful\n");
                  startDataPathArgC.Print ("StartDataPathArgument");
               }
            }
            else {
               printf ("Decode of StartDataPathArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 111 : {  /* stopDataPath operation */
            ASN1T_StopDataPathArgument stopDataPathArg;
            ASN1C_StopDataPathArgument stopDataPathArgC(decodeBuffer2, stopDataPathArg);
            stat = stopDataPathArgC.Decode();
            if (stat == 0) {
               /* process the stopDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of StopDataPathArgument was successful\n");
                  stopDataPathArgC.Print ("StopDataPathArgument");
               }
            }
            else {
               printf ("Decode of StopDataPathArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 112 : {  /* sendData operation */
            ASN1T_SendDataArgument sendDataArg;
            ASN1C_SendDataArgument sendDataArgC(decodeBuffer2, sendDataArg);
            stat = sendDataArgC.Decode();
            if (stat == 0) {
               /* process the sendData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendDataArgument was successful\n");
                  sendDataArgC.Print ("SendDataArgument");
               }
            }
            else {
               printf ("Decode of SendDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 113 : {  /* sendMulticastData operation */
            ASN1T_SendMulticastDataArgument sendMulticastDataArg;
            ASN1C_SendMulticastDataArgument sendMulticastDataArgC(decodeBuffer2, sendMulticastDataArg);
            stat = sendMulticastDataArgC.Decode();
            if (stat == 0) {
               /* process the sendMulticastData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendMulticastDataArgument was successful\n");
                  sendMulticastDataArgC.Print ("SendMulticastDataArgument");
               }
            }
            else {
               printf ("Decode of SendMulticastDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 114 : {  /* sendBroadcastData operation */
            ASN1T_SendBroadcastDataArgument sendBroadcastDataArg;
            ASN1C_SendBroadcastDataArgument sendBroadcastDataArgC(decodeBuffer2, sendBroadcastDataArg);
            stat = sendBroadcastDataArgC.Decode();
            if (stat == 0) {
               /* process the sendBroadcastData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendBroadcastDataArgument was successful\n");
                  sendBroadcastDataArgC.Print ("SendBroadcastDataArgument");
               }
            }
            else {
               printf ("Decode of SendBroadcastDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 115 : {  /* suspendDataPath operation */
            ASN1T_SuspendDataPathArgument suspendDataPathArg;
            ASN1C_SuspendDataPathArgument suspendDataPathArgC(decodeBuffer2, suspendDataPathArg);
            stat = suspendDataPathArgC.Decode();
            if (stat == 0) {
               /* process the suspendDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendDataPathArgument was successful\n");
                  suspendDataPathArgC.Print ("SuspendDataPathArgument");
               }
            }
            else {
               printf ("Decode of SuspendDataPathArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 116 : {  /* dataPathSuspended operation */
            ASN1T_DataPathSuspendedArgument dataPathSuspendedArg;
            ASN1C_DataPathSuspendedArgument dataPathSuspendedArgC(decodeBuffer2, dataPathSuspendedArg);
            stat = dataPathSuspendedArgC.Decode();
            if (stat == 0) {
               /* process the dataPathSuspended operation in your application */
               if (tracearg) {
                  printf ("Decode of DataPathSuspendedArgument was successful\n");
                  dataPathSuspendedArgC.Print ("DataPathSuspendedArgument");
               }
            }
            else {
               printf ("Decode of DataPathSuspendedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 117 : {  /* resumeDataPath operation */
            ASN1T_ResumeDataPathArgument resumeDataPathArg;
            ASN1C_ResumeDataPathArgument resumeDataPathArgC(decodeBuffer2, resumeDataPathArg);
            stat = resumeDataPathArgC.Decode();
            if (stat == 0) {
               /* process the resumeDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeDataPathArgument was successful\n");
                  resumeDataPathArgC.Print ("ResumeDataPathArgument");
               }
            }
            else {
               printf ("Decode of ResumeDataPathArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 118 : {  /* dataPathResumed operation */
            ASN1T_DataPathResumedArgument dataPathResumedArg;
            ASN1C_DataPathResumedArgument dataPathResumedArgC(decodeBuffer2, dataPathResumedArg);
            stat = dataPathResumedArgC.Decode();
            if (stat == 0) {
               /* process the dataPathResumed operation in your application */
               if (tracearg) {
                  printf ("Decode of DataPathResumedArgument was successful\n");
                  dataPathResumedArgC.Print ("DataPathResumedArgument");
               }
            }
            else {
               printf ("Decode of DataPathResumedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 119 : {  /* fastData operation */
            ASN1T_FastDataArgument fastDataArg;
            ASN1C_FastDataArgument fastDataArgC(decodeBuffer2, fastDataArg);
            stat = fastDataArgC.Decode();
            if (stat == 0) {
               /* process the fastData operation in your application */
               if (tracearg) {
                  printf ("Decode of FastDataArgument was successful\n");
                  fastDataArgC.Print ("FastDataArgument");
               }
            }
            else {
               printf ("Decode of FastDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 201 : {  /* getLogicalDeviceInformation operation */
            ASN1T_GetLogicalDeviceInformationArgument getLogicalDeviceInformationArg;
            ASN1C_GetLogicalDeviceInformationArgument getLogicalDeviceInformationArgC(decodeBuffer2, getLogicalDeviceInformationArg);
            stat = getLogicalDeviceInformationArgC.Decode();
            if (stat == 0) {
               /* process the getLogicalDeviceInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLogicalDeviceInformationArgument was successful\n");
                  getLogicalDeviceInformationArgC.Print ("GetLogicalDeviceInformationArgument");
               }
            }
            else {
               printf ("Decode of GetLogicalDeviceInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 202 : {  /* getPhysicalDeviceInformation operation */
            ASN1T_GetPhysicalDeviceInformationArgument getPhysicalDeviceInformationArg;
            ASN1C_GetPhysicalDeviceInformationArgument getPhysicalDeviceInformationArgC(decodeBuffer2, getPhysicalDeviceInformationArg);
            stat = getPhysicalDeviceInformationArgC.Decode();
            if (stat == 0) {
               /* process the getPhysicalDeviceInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetPhysicalDeviceInformationArgument was successful\n");
                  getPhysicalDeviceInformationArgC.Print ("GetPhysicalDeviceInformationArgument");
               }
            }
            else {
               printf ("Decode of GetPhysicalDeviceInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 203 : {  /* getSwitchingFunctionCapabilities operation */
            ASN1T_GetSwitchingFunctionCapsArgument getSwitchingFunctionCapabilitiesArg;
            ASN1C_GetSwitchingFunctionCapsArgument getSwitchingFunctionCapabilitiesArgC(decodeBuffer2, getSwitchingFunctionCapabilitiesArg);
            stat = getSwitchingFunctionCapabilitiesArgC.Decode();
            if (stat == 0) {
               /* process the getSwitchingFunctionCapabilities operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSwitchingFunctionCapsArgument was successful\n");
                  getSwitchingFunctionCapabilitiesArgC.Print ("GetSwitchingFunctionCapsArgument");
               }
            }
            else {
               printf ("Decode of GetSwitchingFunctionCapsArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 204 : {  /* getSwitchingFunctionDevices operation */
            ASN1T_GetSwitchingFunctionDevicesArgument getSwitchingFunctionDevicesArg;
            ASN1C_GetSwitchingFunctionDevicesArgument getSwitchingFunctionDevicesArgC(decodeBuffer2, getSwitchingFunctionDevicesArg);
            stat = getSwitchingFunctionDevicesArgC.Decode();
            if (stat == 0) {
               /* process the getSwitchingFunctionDevices operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSwitchingFunctionDevicesArgument was successful\n");
                  getSwitchingFunctionDevicesArgC.Print ("GetSwitchingFunctionDevicesArgument");
               }
            }
            else {
               printf ("Decode of GetSwitchingFunctionDevicesArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 205 : {  /* switchingFunctionDevices operation */
            ASN1T_SwitchingFunctionDevicesArgument switchingFunctionDevicesArg;
            ASN1C_SwitchingFunctionDevicesArgument switchingFunctionDevicesArgC(decodeBuffer2, switchingFunctionDevicesArg);
            stat = switchingFunctionDevicesArgC.Decode();
            if (stat == 0) {
               /* process the switchingFunctionDevices operation in your application */
               if (tracearg) {
                  printf ("Decode of SwitchingFunctionDevicesArgument was successful\n");
                  switchingFunctionDevicesArgC.Print ("SwitchingFunctionDevicesArgument");
               }
            }
            else {
               printf ("Decode of SwitchingFunctionDevicesArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 206 : {  /* changeSysStatFilter operation */
            ASN1T_ChangeSysStatFilterArg changeSysStatFilterArg;
            ASN1C_ChangeSysStatFilterArg changeSysStatFilterArgC(decodeBuffer2, changeSysStatFilterArg);
            stat = changeSysStatFilterArgC.Decode();
            if (stat == 0) {
               /* process the changeSysStatFilter operation in your application */
               if (tracearg) {
                  printf ("Decode of ChangeSysStatFilterArg was successful\n");
                  changeSysStatFilterArgC.Print ("ChangeSysStatFilterArg");
               }
            }
            else {
               printf ("Decode of ChangeSysStatFilterArg failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 207 : {  /* systemRegister operation */
            ASN1T_SystemRegisterArgument systemRegisterArg;
            ASN1C_SystemRegisterArgument systemRegisterArgC(decodeBuffer2, systemRegisterArg);
            stat = systemRegisterArgC.Decode();
            if (stat == 0) {
               /* process the systemRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemRegisterArgument was successful\n");
                  systemRegisterArgC.Print ("SystemRegisterArgument");
               }
            }
            else {
               printf ("Decode of SystemRegisterArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 208 : {  /* systemRegisterAbort operation */
            ASN1T_SystemRegisterAbortArgument systemRegisterAbortArg;
            ASN1C_SystemRegisterAbortArgument systemRegisterAbortArgC(decodeBuffer2, systemRegisterAbortArg);
            stat = systemRegisterAbortArgC.Decode();
            if (stat == 0) {
               /* process the systemRegisterAbort operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemRegisterAbortArgument was successful\n");
                  systemRegisterAbortArgC.Print ("SystemRegisterAbortArgument");
               }
            }
            else {
               printf ("Decode of SystemRegisterAbortArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 209 : {  /* systemRegisterCancel operation */
            ASN1T_SystemRegisterCancelArgument systemRegisterCancelArg;
            ASN1C_SystemRegisterCancelArgument systemRegisterCancelArgC(decodeBuffer2, systemRegisterCancelArg);
            stat = systemRegisterCancelArgC.Decode();
            if (stat == 0) {
               /* process the systemRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemRegisterCancelArgument was successful\n");
                  systemRegisterCancelArgC.Print ("SystemRegisterCancelArgument");
               }
            }
            else {
               printf ("Decode of SystemRegisterCancelArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 210 : {  /* requestSysStat operation */
            ASN1T_RequestSysStatArg requestSysStatArg;
            ASN1C_RequestSysStatArg requestSysStatArgC(decodeBuffer2, requestSysStatArg);
            stat = requestSysStatArgC.Decode();
            if (stat == 0) {
               /* process the requestSysStat operation in your application */
               if (tracearg) {
                  printf ("Decode of RequestSysStatArg was successful\n");
                  requestSysStatArgC.Print ("RequestSysStatArg");
               }
            }
            else {
               printf ("Decode of RequestSysStatArg failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 211 : {  /* systemStatus operation */
            ASN1T_SystemStatusArg systemStatusArg;
            ASN1C_SystemStatusArg systemStatusArgC(decodeBuffer2, systemStatusArg);
            stat = systemStatusArgC.Decode();
            if (stat == 0) {
               processSystemStatusReq(systemStatusArg);
               if (tracearg) {
                  printf ("Decode of SystemStatusArg was successful\n");
                  systemStatusArgC.Print ("SystemStatusArg");
               }
            }
            else {
               printf ("Decode of SystemStatusArg failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 212 : {  /* swFunctionCapsChanged operation */
            ASN1T_SwFunctionCapsChangedArg swFunctionCapsChangedArg;
            ASN1C_SwFunctionCapsChangedArg swFunctionCapsChangedArgC(decodeBuffer2, swFunctionCapsChangedArg);
            stat = swFunctionCapsChangedArgC.Decode();
            if (stat == 0) {
               /* process the swFunctionCapsChanged operation in your application */
               if (tracearg) {
                  printf ("Decode of SwFunctionCapsChangedArg was successful\n");
                  swFunctionCapsChangedArgC.Print ("SwFunctionCapsChangedArg");
               }
            }
            else {
               printf ("Decode of SwFunctionCapsChangedArg failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 213 : {  /* swFunctionDevicesChanged operation */
            ASN1T_SwFunctionDevicesChangedArg swFunctionDevicesChangedArg;
            ASN1C_SwFunctionDevicesChangedArg swFunctionDevicesChangedArgC(decodeBuffer2, swFunctionDevicesChangedArg);
            stat = swFunctionDevicesChangedArgC.Decode();
            if (stat == 0) {
               /* process the swFunctionDevicesChanged operation in your application */
               if (tracearg) {
                  printf ("Decode of SwFunctionDevicesChangedArg was successful\n");
                  swFunctionDevicesChangedArgC.Print ("SwFunctionDevicesChangedArg");
               }
            }
            else {
               printf ("Decode of SwFunctionDevicesChangedArg failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 214 : {  /* acceptCall operation */
            ASN1T_AcceptCallArgument acceptCallArg;
            ASN1C_AcceptCallArgument acceptCallArgC(decodeBuffer2, acceptCallArg);
            stat = acceptCallArgC.Decode();
            if (stat == 0) {
               /* process the acceptCall operation in your application */
               if (tracearg) {
                  printf ("Decode of AcceptCallArgument was successful\n");
                  acceptCallArgC.Print ("AcceptCallArgument");
               }
            }
            else {
               printf ("Decode of AcceptCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 215 : {  /* callBack operation */
            ASN1T_CallBackArgument callBackArg;
            ASN1C_CallBackArgument callBackArgC(decodeBuffer2, callBackArg);
            stat = callBackArgC.Decode();
            if (stat == 0) {
               /* process the callBack operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackArgument was successful\n");
                  callBackArgC.Print ("CallBackArgument");
               }
            }
            else {
               printf ("Decode of CallBackArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 216 : {  /* callBackMessage operation */
            ASN1T_CallBackMessageArgument callBackMessageArg;
            ASN1C_CallBackMessageArgument callBackMessageArgC(decodeBuffer2, callBackMessageArg);
            stat = callBackMessageArgC.Decode();
            if (stat == 0) {
               /* process the callBackMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackMessageArgument was successful\n");
                  callBackMessageArgC.Print ("CallBackMessageArgument");
               }
            }
            else {
               printf ("Decode of CallBackMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 217 : {  /* camponCall operation */
            ASN1T_CamponCallArgument camponCallArg;
            ASN1C_CamponCallArgument camponCallArgC(decodeBuffer2, camponCallArg);
            stat = camponCallArgC.Decode();
            if (stat == 0) {
               /* process the camponCall operation in your application */
               if (tracearg) {
                  printf ("Decode of CamponCallArgument was successful\n");
                  camponCallArgC.Print ("CamponCallArgument");
               }
            }
            else {
               printf ("Decode of CamponCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 218 : {  /* deflectCall operation */
            ASN1T_DeflectCallArgument deflectCallArg;
            ASN1C_DeflectCallArgument deflectCallArgC(decodeBuffer2, deflectCallArg);
            stat = deflectCallArgC.Decode();
            if (stat == 0) {
               /* process the deflectCall operation in your application */
               if (tracearg) {
                  printf ("Decode of DeflectCallArgument was successful\n");
                  deflectCallArgC.Print ("DeflectCallArgument");
               }
            }
            else {
               printf ("Decode of DeflectCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 219 : {  /* dialDigits operation */
            ASN1T_DialDigitsArgument dialDigitsArg;
            ASN1C_DialDigitsArgument dialDigitsArgC(decodeBuffer2, dialDigitsArg);
            stat = dialDigitsArgC.Decode();
            if (stat == 0) {
               /* process the dialDigits operation in your application */
               if (tracearg) {
                  printf ("Decode of DialDigitsArgument was successful\n");
                  dialDigitsArgC.Print ("DialDigitsArgument");
               }
            }
            else {
               printf ("Decode of DialDigitsArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 220 : {  /* directedPickupCall operation */
            ASN1T_DirectedPickupCallArgument directedPickupCallArg;
            ASN1C_DirectedPickupCallArgument directedPickupCallArgC(decodeBuffer2, directedPickupCallArg);
            stat = directedPickupCallArgC.Decode();
            if (stat == 0) {
               /* process the directedPickupCall operation in your application */
               if (tracearg) {
                  printf ("Decode of DirectedPickupCallArgument was successful\n");
                  directedPickupCallArgC.Print ("DirectedPickupCallArgument");
               }
            }
            else {
               printf ("Decode of DirectedPickupCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 221 : {  /* groupPickupCall operation */
            ASN1T_GroupPickupCallArgument groupPickupCallArg;
            ASN1C_GroupPickupCallArgument groupPickupCallArgC(decodeBuffer2, groupPickupCallArg);
            stat = groupPickupCallArgC.Decode();
            if (stat == 0) {
               /* process the groupPickupCall operation in your application */
               if (tracearg) {
                  printf ("Decode of GroupPickupCallArgument was successful\n");
                  groupPickupCallArgC.Print ("GroupPickupCallArgument");
               }
            }
            else {
               printf ("Decode of GroupPickupCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 222 : {  /* intrudeCall operation */
            ASN1T_IntrudeCallArgument intrudeCallArg;
            ASN1C_IntrudeCallArgument intrudeCallArgC(decodeBuffer2, intrudeCallArg);
            stat = intrudeCallArgC.Decode();
            if (stat == 0) {
               /* process the intrudeCall operation in your application */
               if (tracearg) {
                  printf ("Decode of IntrudeCallArgument was successful\n");
                  intrudeCallArgC.Print ("IntrudeCallArgument");
               }
            }
            else {
               printf ("Decode of IntrudeCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 223 : {  /* joinCall operation */
            ASN1T_JoinCallArgument joinCallArg;
            ASN1C_JoinCallArgument joinCallArgC(decodeBuffer2, joinCallArg);
            stat = joinCallArgC.Decode();
            if (stat == 0) {
               /* process the joinCall operation in your application */
               if (tracearg) {
                  printf ("Decode of JoinCallArgument was successful\n");
                  joinCallArgC.Print ("JoinCallArgument");
               }
            }
            else {
               printf ("Decode of JoinCallArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 230 : {  /* associateData operation */
            ASN1T_AssociateDataArgument associateDataArg;
            ASN1C_AssociateDataArgument associateDataArgC(decodeBuffer2, associateDataArg);
            stat = associateDataArgC.Decode();
            if (stat == 0) {
               /* process the associateData operation in your application */
               if (tracearg) {
                  printf ("Decode of AssociateDataArgument was successful\n");
                  associateDataArgC.Print ("AssociateDataArgument");
               }
            }
            else {
               printf ("Decode of AssociateDataArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 231 : {  /* cancelTelephonyTones operation */
            ASN1T_CancelTelephonyTonesArgument cancelTelephonyTonesArg;
            ASN1C_CancelTelephonyTonesArgument cancelTelephonyTonesArgC(decodeBuffer2, cancelTelephonyTonesArg);
            stat = cancelTelephonyTonesArgC.Decode();
            if (stat == 0) {
               /* process the cancelTelephonyTones operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelTelephonyTonesArgument was successful\n");
                  cancelTelephonyTonesArgC.Print ("CancelTelephonyTonesArgument");
               }
            }
            else {
               printf ("Decode of CancelTelephonyTonesArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 232 : {  /* generateDigits operation */
            ASN1T_GenerateDigitsArgument generateDigitsArg;
            ASN1C_GenerateDigitsArgument generateDigitsArgC(decodeBuffer2, generateDigitsArg);
            stat = generateDigitsArgC.Decode();
            if (stat == 0) {
               /* process the generateDigits operation in your application */
               if (tracearg) {
                  printf ("Decode of GenerateDigitsArgument was successful\n");
                  generateDigitsArgC.Print ("GenerateDigitsArgument");
               }
            }
            else {
               printf ("Decode of GenerateDigitsArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 233 : {  /* generateTelephonyTones operation */
            ASN1T_GenerateTelephonyTonesArgument generateTelephonyTonesArg;
            ASN1C_GenerateTelephonyTonesArgument generateTelephonyTonesArgC(decodeBuffer2, generateTelephonyTonesArg);
            stat = generateTelephonyTonesArgC.Decode();
            if (stat == 0) {
               /* process the generateTelephonyTones operation in your application */
               if (tracearg) {
                  printf ("Decode of GenerateTelephonyTonesArgument was successful\n");
                  generateTelephonyTonesArgC.Print ("GenerateTelephonyTonesArgument");
               }
            }
            else {
               printf ("Decode of GenerateTelephonyTonesArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 234 : {  /* sendUserInfo operation */
            ASN1T_SendUserInfoArgument sendUserInfoArg;
            ASN1C_SendUserInfoArgument sendUserInfoArgC(decodeBuffer2, sendUserInfoArg);
            stat = sendUserInfoArgC.Decode();
            if (stat == 0) {
               /* process the sendUserInfo operation in your application */
               if (tracearg) {
                  printf ("Decode of SendUserInfoArgument was successful\n");
                  sendUserInfoArgC.Print ("SendUserInfoArgument");
               }
            }
            else {
               printf ("Decode of SendUserInfoArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 244 : {  /* attachMediaService operation */
            ASN1T_AttachMediaServiceArgument attachMediaServiceArg;
            ASN1C_AttachMediaServiceArgument attachMediaServiceArgC(decodeBuffer2, attachMediaServiceArg);
            stat = attachMediaServiceArgC.Decode();
            if (stat == 0) {
               /* process the attachMediaService operation in your application */
               if (tracearg) {
                  printf ("Decode of AttachMediaServiceArgument was successful\n");
                  attachMediaServiceArgC.Print ("AttachMediaServiceArgument");
               }
            }
            else {
               printf ("Decode of AttachMediaServiceArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 245 : {  /* detachMediaService operation */
            ASN1T_DetachMediaServiceArgument detachMediaServiceArg;
            ASN1C_DetachMediaServiceArgument detachMediaServiceArgC(decodeBuffer2, detachMediaServiceArg);
            stat = detachMediaServiceArgC.Decode();
            if (stat == 0) {
               /* process the detachMediaService operation in your application */
               if (tracearg) {
                  printf ("Decode of DetachMediaServiceArgument was successful\n");
                  detachMediaServiceArgC.Print ("DetachMediaServiceArgument");
               }
            }
            else {
               printf ("Decode of DetachMediaServiceArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 248 : {  /* routeRegister operation */
            ASN1T_RouteRegisterArgument routeRegisterArg;
            ASN1C_RouteRegisterArgument routeRegisterArgC(decodeBuffer2, routeRegisterArg);
            stat = routeRegisterArgC.Decode();
            if (stat == 0) {
               /* process the routeRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRegisterArgument was successful\n");
                  routeRegisterArgC.Print ("RouteRegisterArgument");
               }
            }
            else {
               printf ("Decode of RouteRegisterArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 249 : {  /* routeRegisterAbort operation */
            ASN1T_RouteRegisterAbortArgument routeRegisterAbortArg;
            ASN1C_RouteRegisterAbortArgument routeRegisterAbortArgC(decodeBuffer2, routeRegisterAbortArg);
            stat = routeRegisterAbortArgC.Decode();
            if (stat == 0) {
               /* process the routeRegisterAbort operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRegisterAbortArgument was successful\n");
                  routeRegisterAbortArgC.Print ("RouteRegisterAbortArgument");
               }
            }
            else {
               printf ("Decode of RouteRegisterAbortArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 250 : {  /* routeRegisterCancel operation */
            ASN1T_RouteRegisterCancelArgument routeRegisterCancelArg;
            ASN1C_RouteRegisterCancelArgument routeRegisterCancelArgC(decodeBuffer2, routeRegisterCancelArg);
            stat = routeRegisterCancelArgC.Decode();
            if (stat == 0) {
               /* process the routeRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRegisterCancelArgument was successful\n");
                  routeRegisterCancelArgC.Print ("RouteRegisterCancelArgument");
               }
            }
            else {
               printf ("Decode of RouteRegisterCancelArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 260 : {  /* buttonPress operation */
            ASN1T_ButtonPressArgument buttonPressArg;
            ASN1C_ButtonPressArgument buttonPressArgC(decodeBuffer2, buttonPressArg);
            stat = buttonPressArgC.Decode();
            if (stat == 0) {
               /* process the buttonPress operation in your application */
               if (tracearg) {
                  printf ("Decode of ButtonPressArgument was successful\n");
                  buttonPressArgC.Print ("ButtonPressArgument");
               }
            }
            else {
               printf ("Decode of ButtonPressArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 261 : {  /* getAuditoryApparatusInformation operation */
            ASN1T_GetAuditoryApparatusInformationArgument getAuditoryApparatusInformationArg;
            ASN1C_GetAuditoryApparatusInformationArgument getAuditoryApparatusInformationArgC(decodeBuffer2, getAuditoryApparatusInformationArg);
            stat = getAuditoryApparatusInformationArgC.Decode();
            if (stat == 0) {
               /* process the getAuditoryApparatusInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAuditoryApparatusInformationArgument was successful\n");
                  getAuditoryApparatusInformationArgC.Print ("GetAuditoryApparatusInformationArgument");
               }
            }
            else {
               printf ("Decode of GetAuditoryApparatusInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 262 : {  /* getButtonInformation operation */
            ASN1T_GetButtonInformationArgument getButtonInformationArg;
            ASN1C_GetButtonInformationArgument getButtonInformationArgC(decodeBuffer2, getButtonInformationArg);
            stat = getButtonInformationArgC.Decode();
            if (stat == 0) {
               /* process the getButtonInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetButtonInformationArgument was successful\n");
                  getButtonInformationArgC.Print ("GetButtonInformationArgument");
               }
            }
            else {
               printf ("Decode of GetButtonInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 263 : {  /* getDisplay operation */
            ASN1T_GetDisplayArgument getDisplayArg;
            ASN1C_GetDisplayArgument getDisplayArgC(decodeBuffer2, getDisplayArg);
            stat = getDisplayArgC.Decode();
            if (stat == 0) {
               /* process the getDisplay operation in your application */
               if (tracearg) {
                  printf ("Decode of GetDisplayArgument was successful\n");
                  getDisplayArgC.Print ("GetDisplayArgument");
               }
            }
            else {
               printf ("Decode of GetDisplayArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 264 : {  /* getHookswitchStatus operation */
            ASN1T_GetHookswitchStatusArgument getHookswitchStatusArg;
            ASN1C_GetHookswitchStatusArgument getHookswitchStatusArgC(decodeBuffer2, getHookswitchStatusArg);
            stat = getHookswitchStatusArgC.Decode();
            if (stat == 0) {
               /* process the getHookswitchStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetHookswitchStatusArgument was successful\n");
                  getHookswitchStatusArgC.Print ("GetHookswitchStatusArgument");
               }
            }
            else {
               printf ("Decode of GetHookswitchStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 265 : {  /* getLampInformation operation */
            ASN1T_GetLampInformationArgument getLampInformationArg;
            ASN1C_GetLampInformationArgument getLampInformationArgC(decodeBuffer2, getLampInformationArg);
            stat = getLampInformationArgC.Decode();
            if (stat == 0) {
               /* process the getLampInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLampInformationArgument was successful\n");
                  getLampInformationArgC.Print ("GetLampInformationArgument");
               }
            }
            else {
               printf ("Decode of GetLampInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 266 : {  /* getLampMode operation */
            ASN1T_GetLampModeArgument getLampModeArg;
            ASN1C_GetLampModeArgument getLampModeArgC(decodeBuffer2, getLampModeArg);
            stat = getLampModeArgC.Decode();
            if (stat == 0) {
               /* process the getLampMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLampModeArgument was successful\n");
                  getLampModeArgC.Print ("GetLampModeArgument");
               }
            }
            else {
               printf ("Decode of GetLampModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 267 : {  /* getMessageWaitingIndicator operation */
            ASN1T_GetMessageWaitingIndicatorArgument getMessageWaitingIndicatorArg;
            ASN1C_GetMessageWaitingIndicatorArgument getMessageWaitingIndicatorArgC(decodeBuffer2, getMessageWaitingIndicatorArg);
            stat = getMessageWaitingIndicatorArgC.Decode();
            if (stat == 0) {
               /* process the getMessageWaitingIndicator operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMessageWaitingIndicatorArgument was successful\n");
                  getMessageWaitingIndicatorArgC.Print ("GetMessageWaitingIndicatorArgument");
               }
            }
            else {
               printf ("Decode of GetMessageWaitingIndicatorArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 268 : {  /* getMicrophoneGain operation */
            ASN1T_GetMicrophoneGainArgument getMicrophoneGainArg;
            ASN1C_GetMicrophoneGainArgument getMicrophoneGainArgC(decodeBuffer2, getMicrophoneGainArg);
            stat = getMicrophoneGainArgC.Decode();
            if (stat == 0) {
               /* process the getMicrophoneGain operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMicrophoneGainArgument was successful\n");
                  getMicrophoneGainArgC.Print ("GetMicrophoneGainArgument");
               }
            }
            else {
               printf ("Decode of GetMicrophoneGainArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 269 : {  /* getMicrophoneMute operation */
            ASN1T_GetMicrophoneMuteArgument getMicrophoneMuteArg;
            ASN1C_GetMicrophoneMuteArgument getMicrophoneMuteArgC(decodeBuffer2, getMicrophoneMuteArg);
            stat = getMicrophoneMuteArgC.Decode();
            if (stat == 0) {
               /* process the getMicrophoneMute operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMicrophoneMuteArgument was successful\n");
                  getMicrophoneMuteArgC.Print ("GetMicrophoneMuteArgument");
               }
            }
            else {
               printf ("Decode of GetMicrophoneMuteArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 270 : {  /* getRingerStatus operation */
            ASN1T_GetRingerStatusArgument getRingerStatusArg;
            ASN1C_GetRingerStatusArgument getRingerStatusArgC(decodeBuffer2, getRingerStatusArg);
            stat = getRingerStatusArgC.Decode();
            if (stat == 0) {
               /* process the getRingerStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetRingerStatusArgument was successful\n");
                  getRingerStatusArgC.Print ("GetRingerStatusArgument");
               }
            }
            else {
               printf ("Decode of GetRingerStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 271 : {  /* getSpeakerMute operation */
            ASN1T_GetSpeakerMuteArgument getSpeakerMuteArg;
            ASN1C_GetSpeakerMuteArgument getSpeakerMuteArgC(decodeBuffer2, getSpeakerMuteArg);
            stat = getSpeakerMuteArgC.Decode();
            if (stat == 0) {
               /* process the getSpeakerMute operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSpeakerMuteArgument was successful\n");
                  getSpeakerMuteArgC.Print ("GetSpeakerMuteArgument");
               }
            }
            else {
               printf ("Decode of GetSpeakerMuteArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 272 : {  /* getSpeakerVolume operation */
            ASN1T_GetSpeakerVolumeArgument getSpeakerVolumeArg;
            ASN1C_GetSpeakerVolumeArgument getSpeakerVolumeArgC(decodeBuffer2, getSpeakerVolumeArg);
            stat = getSpeakerVolumeArgC.Decode();
            if (stat == 0) {
               /* process the getSpeakerVolume operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSpeakerVolumeArgument was successful\n");
                  getSpeakerVolumeArgC.Print ("GetSpeakerVolumeArgument");
               }
            }
            else {
               printf ("Decode of GetSpeakerVolumeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 273 : {  /* setButtonInformation operation */
            ASN1T_SetButtonInformationArgument setButtonInformationArg;
            ASN1C_SetButtonInformationArgument setButtonInformationArgC(decodeBuffer2, setButtonInformationArg);
            stat = setButtonInformationArgC.Decode();
            if (stat == 0) {
               /* process the setButtonInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of SetButtonInformationArgument was successful\n");
                  setButtonInformationArgC.Print ("SetButtonInformationArgument");
               }
            }
            else {
               printf ("Decode of SetButtonInformationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 274 : {  /* setDisplay operation */
            ASN1T_SetDisplayArgument setDisplayArg;
            ASN1C_SetDisplayArgument setDisplayArgC(decodeBuffer2, setDisplayArg);
            stat = setDisplayArgC.Decode();
            if (stat == 0) {
               /* process the setDisplay operation in your application */
               if (tracearg) {
                  printf ("Decode of SetDisplayArgument was successful\n");
                  setDisplayArgC.Print ("SetDisplayArgument");
               }
            }
            else {
               printf ("Decode of SetDisplayArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 275 : {  /* setHookswitchStatus operation */
            ASN1T_SetHookswitchStatusArgument setHookswitchStatusArg;
            ASN1C_SetHookswitchStatusArgument setHookswitchStatusArgC(decodeBuffer2, setHookswitchStatusArg);
            stat = setHookswitchStatusArgC.Decode();
            if (stat == 0) {
               /* process the setHookswitchStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetHookswitchStatusArgument was successful\n");
                  setHookswitchStatusArgC.Print ("SetHookswitchStatusArgument");
               }
            }
            else {
               printf ("Decode of SetHookswitchStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 276 : {  /* setLampMode operation */
            ASN1T_SetLampModeArgument setLampModeArg;
            ASN1C_SetLampModeArgument setLampModeArgC(decodeBuffer2, setLampModeArg);
            stat = setLampModeArgC.Decode();
            if (stat == 0) {
               /* process the setLampMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetLampModeArgument was successful\n");
                  setLampModeArgC.Print ("SetLampModeArgument");
               }
            }
            else {
               printf ("Decode of SetLampModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 277 : {  /* setMessageWaitingIndicator operation */
            ASN1T_SetMessageWaitingIndicatorArgument setMessageWaitingIndicatorArg;
            ASN1C_SetMessageWaitingIndicatorArgument setMessageWaitingIndicatorArgC(decodeBuffer2, setMessageWaitingIndicatorArg);
            stat = setMessageWaitingIndicatorArgC.Decode();
            if (stat == 0) {
               /* process the setMessageWaitingIndicator operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMessageWaitingIndicatorArgument was successful\n");
                  setMessageWaitingIndicatorArgC.Print ("SetMessageWaitingIndicatorArgument");
               }
            }
            else {
               printf ("Decode of SetMessageWaitingIndicatorArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 278 : {  /* setMicrophoneGain operation */
            ASN1T_SetMicrophoneGainArgument setMicrophoneGainArg;
            ASN1C_SetMicrophoneGainArgument setMicrophoneGainArgC(decodeBuffer2, setMicrophoneGainArg);
            stat = setMicrophoneGainArgC.Decode();
            if (stat == 0) {
               /* process the setMicrophoneGain operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMicrophoneGainArgument was successful\n");
                  setMicrophoneGainArgC.Print ("SetMicrophoneGainArgument");
               }
            }
            else {
               printf ("Decode of SetMicrophoneGainArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 279 : {  /* setMicrophoneMute operation */
            ASN1T_SetMicrophoneMuteArgument setMicrophoneMuteArg;
            ASN1C_SetMicrophoneMuteArgument setMicrophoneMuteArgC(decodeBuffer2, setMicrophoneMuteArg);
            stat = setMicrophoneMuteArgC.Decode();
            if (stat == 0) {
               /* process the setMicrophoneMute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMicrophoneMuteArgument was successful\n");
                  setMicrophoneMuteArgC.Print ("SetMicrophoneMuteArgument");
               }
            }
            else {
               printf ("Decode of SetMicrophoneMuteArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 280 : {  /* setRingerStatus operation */
            ASN1T_SetRingerStatusArgument setRingerStatusArg;
            ASN1C_SetRingerStatusArgument setRingerStatusArgC(decodeBuffer2, setRingerStatusArg);
            stat = setRingerStatusArgC.Decode();
            if (stat == 0) {
               /* process the setRingerStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetRingerStatusArgument was successful\n");
                  setRingerStatusArgC.Print ("SetRingerStatusArgument");
               }
            }
            else {
               printf ("Decode of SetRingerStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 281 : {  /* setSpeakerMute operation */
            ASN1T_SetSpeakerMuteArgument setSpeakerMuteArg;
            ASN1C_SetSpeakerMuteArgument setSpeakerMuteArgC(decodeBuffer2, setSpeakerMuteArg);
            stat = setSpeakerMuteArgC.Decode();
            if (stat == 0) {
               /* process the setSpeakerMute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetSpeakerMuteArgument was successful\n");
                  setSpeakerMuteArgC.Print ("SetSpeakerMuteArgument");
               }
            }
            else {
               printf ("Decode of SetSpeakerMuteArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 282 : {  /* setSpeakerVolume operation */
            ASN1T_SetSpeakerVolumeArgument setSpeakerVolumeArg;
            ASN1C_SetSpeakerVolumeArgument setSpeakerVolumeArgC(decodeBuffer2, setSpeakerVolumeArg);
            stat = setSpeakerVolumeArgC.Decode();
            if (stat == 0) {
               /* process the setSpeakerVolume operation in your application */
               if (tracearg) {
                  printf ("Decode of SetSpeakerVolumeArgument was successful\n");
                  setSpeakerVolumeArgC.Print ("SetSpeakerVolumeArgument");
               }
            }
            else {
               printf ("Decode of SetSpeakerVolumeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 300 : {  /* callBackNonCallRelated operation */
            ASN1T_CallBackNonCallRelatedArgument callBackNonCallRelatedArg;
            ASN1C_CallBackNonCallRelatedArgument callBackNonCallRelatedArgC(decodeBuffer2, callBackNonCallRelatedArg);
            stat = callBackNonCallRelatedArgC.Decode();
            if (stat == 0) {
               /* process the callBackNonCallRelated operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackNonCallRelatedArgument was successful\n");
                  callBackNonCallRelatedArgC.Print ("CallBackNonCallRelatedArgument");
               }
            }
            else {
               printf ("Decode of CallBackNonCallRelatedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 301 : {  /* callBackMessageNonCallRelated operation */
            ASN1T_CallBackMessageNonCallRelatedArgument callBackMessageNonCallRelatedArg;
            ASN1C_CallBackMessageNonCallRelatedArgument callBackMessageNonCallRelatedArgC(decodeBuffer2, callBackMessageNonCallRelatedArg);
            stat = callBackMessageNonCallRelatedArgC.Decode();
            if (stat == 0) {
               /* process the callBackMessageNonCallRelated operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackMessageNonCallRelatedArgument was successful\n");
                  callBackMessageNonCallRelatedArgC.Print ("CallBackMessageNonCallRelatedArgument");
               }
            }
            else {
               printf ("Decode of CallBackMessageNonCallRelatedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 302 : {  /* cancelCallBack operation */
            ASN1T_CancelCallBackArgument cancelCallBackArg;
            ASN1C_CancelCallBackArgument cancelCallBackArgC(decodeBuffer2, cancelCallBackArg);
            stat = cancelCallBackArgC.Decode();
            if (stat == 0) {
               /* process the cancelCallBack operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelCallBackArgument was successful\n");
                  cancelCallBackArgC.Print ("CancelCallBackArgument");
               }
            }
            else {
               printf ("Decode of CancelCallBackArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 303 : {  /* cancelCallBackMessage operation */
            ASN1T_CancelCallBackMessageArgument cancelCallBackMessageArg;
            ASN1C_CancelCallBackMessageArgument cancelCallBackMessageArgC(decodeBuffer2, cancelCallBackMessageArg);
            stat = cancelCallBackMessageArgC.Decode();
            if (stat == 0) {
               /* process the cancelCallBackMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelCallBackMessageArgument was successful\n");
                  cancelCallBackMessageArgC.Print ("CancelCallBackMessageArgument");
               }
            }
            else {
               printf ("Decode of CancelCallBackMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 304 : {  /* getAgentState operation */
            ASN1T_GetAgentStateArgument getAgentStateArg;
            ASN1C_GetAgentStateArgument getAgentStateArgC(decodeBuffer2, getAgentStateArg);
            stat = getAgentStateArgC.Decode();
            if (stat == 0) {
               /* process the getAgentState operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAgentStateArgument was successful\n");
                  getAgentStateArgC.Print ("GetAgentStateArgument");
               }
            }
            else {
               printf ("Decode of GetAgentStateArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 305 : {  /* getAutoAnswer operation */
            ASN1T_GetAutoAnswerArgument getAutoAnswerArg;
            ASN1C_GetAutoAnswerArgument getAutoAnswerArgC(decodeBuffer2, getAutoAnswerArg);
            stat = getAutoAnswerArgC.Decode();
            if (stat == 0) {
               /* process the getAutoAnswer operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAutoAnswerArgument was successful\n");
                  getAutoAnswerArgC.Print ("GetAutoAnswerArgument");
               }
            }
            else {
               printf ("Decode of GetAutoAnswerArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 306 : {  /* getAutoWorkMode operation */
            ASN1T_GetAutoWorkModeArgument getAutoWorkModeArg;
            ASN1C_GetAutoWorkModeArgument getAutoWorkModeArgC(decodeBuffer2, getAutoWorkModeArg);
            stat = getAutoWorkModeArgC.Decode();
            if (stat == 0) {
               /* process the getAutoWorkMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAutoWorkModeArgument was successful\n");
                  getAutoWorkModeArgC.Print ("GetAutoWorkModeArgument");
               }
            }
            else {
               printf ("Decode of GetAutoWorkModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 307 : {  /* getCallerIDStatus operation */
            ASN1T_GetCallerIDStatusArgument getCallerIDStatusArg;
            ASN1C_GetCallerIDStatusArgument getCallerIDStatusArgC(decodeBuffer2, getCallerIDStatusArg);
            stat = getCallerIDStatusArgC.Decode();
            if (stat == 0) {
               /* process the getCallerIDStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetCallerIDStatusArgument was successful\n");
                  getCallerIDStatusArgC.Print ("GetCallerIDStatusArgument");
               }
            }
            else {
               printf ("Decode of GetCallerIDStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 308 : {  /* getDoNotDisturb operation */
            ASN1T_GetDoNotDisturbArgument getDoNotDisturbArg;
            ASN1C_GetDoNotDisturbArgument getDoNotDisturbArgC(decodeBuffer2, getDoNotDisturbArg);
            stat = getDoNotDisturbArgC.Decode();
            if (stat == 0) {
               /* process the getDoNotDisturb operation in your application */
               if (tracearg) {
                  printf ("Decode of GetDoNotDisturbArgument was successful\n");
                  getDoNotDisturbArgC.Print ("GetDoNotDisturbArgument");
               }
            }
            else {
               printf ("Decode of GetDoNotDisturbArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 309 : {  /* getForwarding operation */
            ASN1T_GetForwardingArgument getForwardingArg;
            ASN1C_GetForwardingArgument getForwardingArgC(decodeBuffer2, getForwardingArg);
            stat = getForwardingArgC.Decode();
            if (stat == 0) {
               /* process the getForwarding operation in your application */
               if (tracearg) {
                  printf ("Decode of GetForwardingArgument was successful\n");
                  getForwardingArgC.Print ("GetForwardingArgument");
               }
            }
            else {
               printf ("Decode of GetForwardingArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 310 : {  /* getLastNumberDialed operation */
            ASN1T_GetLastNumberDialedArgument getLastNumberDialedArg;
            ASN1C_GetLastNumberDialedArgument getLastNumberDialedArgC(decodeBuffer2, getLastNumberDialedArg);
            stat = getLastNumberDialedArgC.Decode();
            if (stat == 0) {
               /* process the getLastNumberDialed operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLastNumberDialedArgument was successful\n");
                  getLastNumberDialedArgC.Print ("GetLastNumberDialedArgument");
               }
            }
            else {
               printf ("Decode of GetLastNumberDialedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 311 : {  /* getRouteingMode operation */
            ASN1T_GetRouteingModeArgument getRouteingModeArg;
            ASN1C_GetRouteingModeArgument getRouteingModeArgC(decodeBuffer2, getRouteingModeArg);
            stat = getRouteingModeArgC.Decode();
            if (stat == 0) {
               /* process the getRouteingMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetRouteingModeArgument was successful\n");
                  getRouteingModeArgC.Print ("GetRouteingModeArgument");
               }
            }
            else {
               printf ("Decode of GetRouteingModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 312 : {  /* setAgentState operation */
            ASN1T_SetAgentStateArgument setAgentStateArg;
            ASN1C_SetAgentStateArgument setAgentStateArgC(decodeBuffer2, setAgentStateArg);
            stat = setAgentStateArgC.Decode();
            if (stat == 0) {
               /* process the setAgentState operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAgentStateArgument was successful\n");
                  setAgentStateArgC.Print ("SetAgentStateArgument");
               }
            }
            else {
               printf ("Decode of SetAgentStateArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 313 : {  /* setAutoAnswer operation */
            ASN1T_SetAutoAnswerArgument setAutoAnswerArg;
            ASN1C_SetAutoAnswerArgument setAutoAnswerArgC(decodeBuffer2, setAutoAnswerArg);
            stat = setAutoAnswerArgC.Decode();
            if (stat == 0) {
               /* process the setAutoAnswer operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAutoAnswerArgument was successful\n");
                  setAutoAnswerArgC.Print ("SetAutoAnswerArgument");
               }
            }
            else {
               printf ("Decode of SetAutoAnswerArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 314 : {  /* setAutoWorkMode operation */
            ASN1T_SetAutoWorkModeArgument setAutoWorkModeArg;
            ASN1C_SetAutoWorkModeArgument setAutoWorkModeArgC(decodeBuffer2, setAutoWorkModeArg);
            stat = setAutoWorkModeArgC.Decode();
            if (stat == 0) {
               /* process the setAutoWorkMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAutoWorkModeArgument was successful\n");
                  setAutoWorkModeArgC.Print ("SetAutoWorkModeArgument");
               }
            }
            else {
               printf ("Decode of SetAutoWorkModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 315 : {  /* setCallerIDStatus operation */
            ASN1T_SetCallerIDStatusArgument setCallerIDStatusArg;
            ASN1C_SetCallerIDStatusArgument setCallerIDStatusArgC(decodeBuffer2, setCallerIDStatusArg);
            stat = setCallerIDStatusArgC.Decode();
            if (stat == 0) {
               /* process the setCallerIDStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetCallerIDStatusArgument was successful\n");
                  setCallerIDStatusArgC.Print ("SetCallerIDStatusArgument");
               }
            }
            else {
               printf ("Decode of SetCallerIDStatusArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 316 : {  /* setDoNotDisturb operation */
            ASN1T_SetDoNotDisturbArgument setDoNotDisturbArg;
            ASN1C_SetDoNotDisturbArgument setDoNotDisturbArgC(decodeBuffer2, setDoNotDisturbArg);
            stat = setDoNotDisturbArgC.Decode();
            if (stat == 0) {
               /* process the setDoNotDisturb operation in your application */
               if (tracearg) {
                  printf ("Decode of SetDoNotDisturbArgument was successful\n");
                  setDoNotDisturbArgC.Print ("SetDoNotDisturbArgument");
               }
            }
            else {
               printf ("Decode of SetDoNotDisturbArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 317 : {  /* setForwarding operation */
            ASN1T_SetForwardingArgument setForwardingArg;
            ASN1C_SetForwardingArgument setForwardingArgC(decodeBuffer2, setForwardingArg);
            stat = setForwardingArgC.Decode();
            if (stat == 0) {
               /* process the setForwarding operation in your application */
               if (tracearg) {
                  printf ("Decode of SetForwardingArgument was successful\n");
                  setForwardingArgC.Print ("SetForwardingArgument");
               }
            }
            else {
               printf ("Decode of SetForwardingArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 318 : {  /* setRouteingMode operation */
            ASN1T_SetRouteingModeArgument setRouteingModeArg;
            ASN1C_SetRouteingModeArgument setRouteingModeArgC(decodeBuffer2, setRouteingModeArg);
            stat = setRouteingModeArgC.Decode();
            if (stat == 0) {
               /* process the setRouteingMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetRouteingModeArgument was successful\n");
                  setRouteingModeArgC.Print ("SetRouteingModeArgument");
               }
            }
            else {
               printf ("Decode of SetRouteingModeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 340 : {  /* ioRegister operation */
            ASN1T_IORegisterArgument ioRegisterArg;
            ASN1C_IORegisterArgument ioRegisterArgC(decodeBuffer2, ioRegisterArg);
            stat = ioRegisterArgC.Decode();
            if (stat == 0) {
               /* process the ioRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of IORegisterArgument was successful\n");
                  ioRegisterArgC.Print ("IORegisterArgument");
               }
            }
            else {
               printf ("Decode of IORegisterArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 341 : {  /* ioRegisterAbort operation */
            ASN1T_IORegisterAbortArgument ioRegisterAbortArg;
            ASN1C_IORegisterAbortArgument ioRegisterAbortArgC(decodeBuffer2, ioRegisterAbortArg);
            stat = ioRegisterAbortArgC.Decode();
            if (stat == 0) {
               /* process the ioRegisterAbort operation in your application */
               if (tracearg) {
                  printf ("Decode of IORegisterAbortArgument was successful\n");
                  ioRegisterAbortArgC.Print ("IORegisterAbortArgument");
               }
            }
            else {
               printf ("Decode of IORegisterAbortArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 342 : {  /* ioRegisterCancel operation */
            ASN1T_IORegisterCancelArgument ioRegisterCancelArg;
            ASN1C_IORegisterCancelArgument ioRegisterCancelArgC(decodeBuffer2, ioRegisterCancelArg);
            stat = ioRegisterCancelArgC.Decode();
            if (stat == 0) {
               /* process the ioRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of IORegisterCancelArgument was successful\n");
                  ioRegisterCancelArgC.Print ("IORegisterCancelArgument");
               }
            }
            else {
               printf ("Decode of IORegisterCancelArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;
      case 343 : {  /* dataCollected operation */
            ASN1T_DataCollectedArgument dataCollectedArg;
            ASN1C_DataCollectedArgument dataCollectedArgC(decodeBuffer2, dataCollectedArg);
            stat = dataCollectedArgC.Decode();
            if (stat == 0) {
               /* process the dataCollected operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectedArgument was successful\n");
                  dataCollectedArgC.Print ("DataCollectedArgument");
               }
            }
            else {
               printf ("Decode of DataCollectedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 344 : {  /* dataCollectionResumed operation */
            ASN1T_DataCollectionResumedArgument dataCollectionResumedArg;
            ASN1C_DataCollectionResumedArgument dataCollectionResumedArgC(decodeBuffer2, dataCollectionResumedArg);
            stat = dataCollectionResumedArgC.Decode();
            if (stat == 0) {
               /* process the dataCollectionResumed operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectionResumedArgument was successful\n");
                  dataCollectionResumedArgC.Print ("DataCollectionResumedArgument");
               }
            }
            else {
               printf ("Decode of DataCollectionResumedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 345 : {  /* dataCollectionSuspended operation */
            ASN1T_DataCollectionSuspendedArgument dataCollectionSuspendedArg;
            ASN1C_DataCollectionSuspendedArgument dataCollectionSuspendedArgC(decodeBuffer2, dataCollectionSuspendedArg);
            stat = dataCollectionSuspendedArgC.Decode();
            if (stat == 0) {
               /* process the dataCollectionSuspended operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectionSuspendedArgument was successful\n");
                  dataCollectionSuspendedArgC.Print ("DataCollectionSuspendedArgument");
               }
            }
            else {
               printf ("Decode of DataCollectionSuspendedArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 346 : {  /* resumeDataCollection operation */
            ASN1T_ResumeDataCollectionArgument resumeDataCollectionArg;
            ASN1C_ResumeDataCollectionArgument resumeDataCollectionArgC(decodeBuffer2, resumeDataCollectionArg);
            stat = resumeDataCollectionArgC.Decode();
            if (stat == 0) {
               /* process the resumeDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeDataCollectionArgument was successful\n");
                  resumeDataCollectionArgC.Print ("ResumeDataCollectionArgument");
               }
            }
            else {
               printf ("Decode of ResumeDataCollectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 347 : {  /* startDataCollection operation */
            ASN1T_StartDataCollectionArgument startDataCollectionArg;
            ASN1C_StartDataCollectionArgument startDataCollectionArgC(decodeBuffer2, startDataCollectionArg);
            stat = startDataCollectionArgC.Decode();
            if (stat == 0) {
               /* process the startDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of StartDataCollectionArgument was successful\n");
                  startDataCollectionArgC.Print ("StartDataCollectionArgument");
               }
            }
            else {
               printf ("Decode of StartDataCollectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 348 : {  /* stopDataCollection operation */
            ASN1T_StopDataCollectionArgument stopDataCollectionArg;
            ASN1C_StopDataCollectionArgument stopDataCollectionArgC(decodeBuffer2, stopDataCollectionArg);
            stat = stopDataCollectionArgC.Decode();
            if (stat == 0) {
               /* process the stopDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of StopDataCollectionArgument was successful\n");
                  stopDataCollectionArgC.Print ("StopDataCollectionArgument");
               }
            }
            else {
               printf ("Decode of StopDataCollectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 349 : {  /* suspendDataCollection operation */
            ASN1T_SuspendDataCollectionArgument suspendDataCollectionArg;
            ASN1C_SuspendDataCollectionArgument suspendDataCollectionArgC(decodeBuffer2, suspendDataCollectionArg);
            stat = suspendDataCollectionArgC.Decode();
            if (stat == 0) {
               /* process the suspendDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendDataCollectionArgument was successful\n");
                  suspendDataCollectionArgC.Print ("SuspendDataCollectionArgument");
               }
            }
            else {
               printf ("Decode of SuspendDataCollectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 500 : {  /* concatenateMessage operation */
            ASN1T_ConcatenateMessageArgument concatenateMessageArg;
            ASN1C_ConcatenateMessageArgument concatenateMessageArgC(decodeBuffer2, concatenateMessageArg);
            stat = concatenateMessageArgC.Decode();
            if (stat == 0) {
               /* process the concatenateMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of ConcatenateMessageArgument was successful\n");
                  concatenateMessageArgC.Print ("ConcatenateMessageArgument");
               }
            }
            else {
               printf ("Decode of ConcatenateMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 501 : {  /* deleteMessage operation */
            ASN1T_DeleteMessageArgument deleteMessageArg;
            ASN1C_DeleteMessageArgument deleteMessageArgC(decodeBuffer2, deleteMessageArg);
            stat = deleteMessageArgC.Decode();
            if (stat == 0) {
               /* process the deleteMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of DeleteMessageArgument was successful\n");
                  deleteMessageArgC.Print ("DeleteMessageArgument");
               }
            }
            else {
               printf ("Decode of DeleteMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 502 : {  /* playMessage operation */
            ASN1T_PlayMessageArgument playMessageArg;
            ASN1C_PlayMessageArgument playMessageArgC(decodeBuffer2, playMessageArg);
            stat = playMessageArgC.Decode();
            if (stat == 0) {
               /* process the playMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of PlayMessageArgument was successful\n");
                  playMessageArgC.Print ("PlayMessageArgument");
               }
            }
            else {
               printf ("Decode of PlayMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 503 : {  /* queryVoiceAttribute operation */
            ASN1T_QueryVoiceAttributeArgument queryVoiceAttributeArg;
            ASN1C_QueryVoiceAttributeArgument queryVoiceAttributeArgC(decodeBuffer2, queryVoiceAttributeArg);
            stat = queryVoiceAttributeArgC.Decode();
            if (stat == 0) {
               /* process the queryVoiceAttribute operation in your application */
               if (tracearg) {
                  printf ("Decode of QueryVoiceAttributeArgument was successful\n");
                  queryVoiceAttributeArgC.Print ("QueryVoiceAttributeArgument");
               }
            }
            else {
               printf ("Decode of QueryVoiceAttributeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 511 : {  /* recordMessage operation */
            ASN1T_RecordMessageArgument recordMessageArg;
            ASN1C_RecordMessageArgument recordMessageArgC(decodeBuffer2, recordMessageArg);
            stat = recordMessageArgC.Decode();
            if (stat == 0) {
               /* process the recordMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of RecordMessageArgument was successful\n");
                  recordMessageArgC.Print ("RecordMessageArgument");
               }
            }
            else {
               printf ("Decode of RecordMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 504 : {  /* reposition operation */
            ASN1T_RepositionArgument repositionArg;
            ASN1C_RepositionArgument repositionArgC(decodeBuffer2, repositionArg);
            stat = repositionArgC.Decode();
            if (stat == 0) {
               /* process the reposition operation in your application */
               if (tracearg) {
                  printf ("Decode of RepositionArgument was successful\n");
                  repositionArgC.Print ("RepositionArgument");
               }
            }
            else {
               printf ("Decode of RepositionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 505 : {  /* resume operation */
            ASN1T_ResumeArgument resumeArg;
            ASN1C_ResumeArgument resumeArgC(decodeBuffer2, resumeArg);
            stat = resumeArgC.Decode();
            if (stat == 0) {
               /* process the resume operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeArgument was successful\n");
                  resumeArgC.Print ("ResumeArgument");
               }
            }
            else {
               printf ("Decode of ResumeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 506 : {  /* review operation */
            ASN1T_ReviewArgument reviewArg;
            ASN1C_ReviewArgument reviewArgC(decodeBuffer2, reviewArg);
            stat = reviewArgC.Decode();
            if (stat == 0) {
               /* process the review operation in your application */
               if (tracearg) {
                  printf ("Decode of ReviewArgument was successful\n");
                  reviewArgC.Print ("ReviewArgument");
               }
            }
            else {
               printf ("Decode of ReviewArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 507 : {  /* setVoiceAttribute operation */
            ASN1T_SetVoiceAttributeArgument setVoiceAttributeArg;
            ASN1C_SetVoiceAttributeArgument setVoiceAttributeArgC(decodeBuffer2, setVoiceAttributeArg);
            stat = setVoiceAttributeArgC.Decode();
            if (stat == 0) {
               /* process the setVoiceAttribute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetVoiceAttributeArgument was successful\n");
                  setVoiceAttributeArgC.Print ("SetVoiceAttributeArgument");
               }
            }
            else {
               printf ("Decode of SetVoiceAttributeArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 508 : {  /* stop operation */
            ASN1T_StopArgument stopArg;
            ASN1C_StopArgument stopArgC(decodeBuffer2, stopArg);
            stat = stopArgC.Decode();
            if (stat == 0) {
               /* process the stop operation in your application */
               if (tracearg) {
                  printf ("Decode of StopArgument was successful\n");
                  stopArgC.Print ("StopArgument");
               }
            }
            else {
               printf ("Decode of StopArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 509 : {  /* suspend operation */
            ASN1T_SuspendArgument suspendArg;
            ASN1C_SuspendArgument suspendArgC(decodeBuffer2, suspendArg);
            stat = suspendArgC.Decode();
            if (stat == 0) {
               /* process the suspend operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendArgument was successful\n");
                  suspendArgC.Print ("SuspendArgument");
               }
            }
            else {
               printf ("Decode of SuspendArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 510 : {  /* synthesizeMessage operation */
            ASN1T_SynthesizeMessageArgument synthesizeMessageArg;
            ASN1C_SynthesizeMessageArgument synthesizeMessageArgC(decodeBuffer2, synthesizeMessageArg);
            stat = synthesizeMessageArgC.Decode();
            if (stat == 0) {
               /* process the synthesizeMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of SynthesizeMessageArgument was successful\n");
                  synthesizeMessageArgC.Print ("SynthesizeMessageArgument");
               }
            }
            else {
               printf ("Decode of SynthesizeMessageArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      case 360 : {  /* cDRNotification operation */
            ASN1T_CDRNotificationArgument cDRNotificationArg;
            ASN1C_CDRNotificationArgument cDRNotificationArgC(decodeBuffer2, cDRNotificationArg);
            stat = cDRNotificationArgC.Decode();
            if (stat == 0) {
               /* process the cDRNotification operation in your application */
               if (tracearg) {
                  printf ("Decode of CDRNotificationArgument was successful\n");
                  cDRNotificationArgC.Print ("CDRNotificationArgument");
               }
            }
            else {
               printf ("Decode of CDRNotificationArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 361 : {  /* cDRReport operation */
            ASN1T_CDRReportArgument cDRReportArg;
            ASN1C_CDRReportArgument cDRReportArgC(decodeBuffer2, cDRReportArg);
            stat = cDRReportArgC.Decode();
            if (stat == 0) {
               /* process the cDRReport operation in your application */
               if (tracearg) {
                  printf ("Decode of CDRReportArgument was successful\n");
                  cDRReportArgC.Print ("CDRReportArgument");
               }
            }
            else {
               printf ("Decode of CDRReportArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 362 : {  /* sendStoredCDR operation */
            ASN1T_SendStoredCDRArgument sendStoredCDRArg;
            ASN1C_SendStoredCDRArgument sendStoredCDRArgC(decodeBuffer2, sendStoredCDRArg);
            stat = sendStoredCDRArgC.Decode();
            if (stat == 0) {
               /* process the sendStoredCDR operation in your application */
               if (tracearg) {
                  printf ("Decode of SendStoredCDRArgument was successful\n");
                  sendStoredCDRArgC.Print ("SendStoredCDRArgument");
               }
            }
            else {
               printf ("Decode of SendStoredCDRArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 363 : {  /* startCDRTransmission operation */
            ASN1T_StartCDRTransmissionArgument startCDRTransmissionArg;
            ASN1C_StartCDRTransmissionArgument startCDRTransmissionArgC(decodeBuffer2, startCDRTransmissionArg);
            stat = startCDRTransmissionArgC.Decode();
            if (stat == 0) {
               /* process the startCDRTransmission operation in your application */
               if (tracearg) {
                  printf ("Decode of StartCDRTransmissionArgument was successful\n");
                  startCDRTransmissionArgC.Print ("StartCDRTransmissionArgument");
               }
            }
            else {
               printf ("Decode of StartCDRTransmissionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 364 : {  /* stopCDRTransmission operation */
            ASN1T_StopCDRTransmissionArgument stopCDRTransmissionArg;
            ASN1C_StopCDRTransmissionArgument stopCDRTransmissionArgC(decodeBuffer2, stopCDRTransmissionArg);
            stat = stopCDRTransmissionArgC.Decode();
            if (stat == 0) {
               /* process the stopCDRTransmission operation in your application */
               if (tracearg) {
                  printf ("Decode of StopCDRTransmissionArgument was successful\n");
                  stopCDRTransmissionArgC.Print ("StopCDRTransmissionArgument");
               }
            }
            else {
               printf ("Decode of StopCDRTransmissionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 365 : {  /* escapeRegister operation */
            ASN1T_EscapeRegisterArgument escapeRegisterArg;
            ASN1C_EscapeRegisterArgument escapeRegisterArgC(decodeBuffer2, escapeRegisterArg);
            stat = escapeRegisterArgC.Decode();
            if (stat == 0) {
               /* process the escapeRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeRegisterArgument was successful\n");
                  escapeRegisterArgC.Print ("EscapeRegisterArgument");
               }
            }
            else {
               printf ("Decode of EscapeRegisterArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 366 : {  /* escapeRegisterAbort operation */
            ASN1T_EscapeRegisterAbortArgument escapeRegisterAbortArg;
            ASN1C_EscapeRegisterAbortArgument escapeRegisterAbortArgC(decodeBuffer2, escapeRegisterAbortArg);
            stat = escapeRegisterAbortArgC.Decode();
            if (stat == 0) {
               /* process the escapeRegisterAbort operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeRegisterAbortArgument was successful\n");
                  escapeRegisterAbortArgC.Print ("EscapeRegisterAbortArgument");
               }
            }
            else {
               printf ("Decode of EscapeRegisterAbortArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 367 : {  /* escapeRegisterCancel operation */
            ASN1T_EscapeRegisterCancelArgument escapeRegisterCancelArg;
            ASN1C_EscapeRegisterCancelArgument escapeRegisterCancelArgC(decodeBuffer2, escapeRegisterCancelArg);
            stat = escapeRegisterCancelArgC.Decode();
            if (stat == 0) {
               /* process the escapeRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeRegisterCancelArgument was successful\n");
                  escapeRegisterCancelArgC.Print ("EscapeRegisterCancelArgument");
               }
            }
            else {
               printf ("Decode of EscapeRegisterCancelArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 368 : {  /* privateDataVersionSelection operation */
            ASN1T_PrivateDataVersionSelectionArgument privateDataVersionSelectionArg;
            ASN1C_PrivateDataVersionSelectionArgument privateDataVersionSelectionArgC(decodeBuffer2, privateDataVersionSelectionArg);
            stat = privateDataVersionSelectionArgC.Decode();
            if (stat == 0) {
               /* process the privateDataVersionSelection operation in your application */
               if (tracearg) {
                  printf ("Decode of PrivateDataVersionSelectionArgument was successful\n");
                  privateDataVersionSelectionArgC.Print ("PrivateDataVersionSelectionArgument");
               }
            }
            else {
               printf ("Decode of PrivateDataVersionSelectionArgument failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      default: {
         printf ("error: invalid operation code, not defined in csta phase 3\n");
         } break;
   }
}

void Client::processEvent(ASN1T_cerd_CSTAEventReportArgument& eventReportArg, OSBOOL tracearg)
{
   //Process the event from EventReportArgument type

}

void Client::processReturnResult(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL tracearg)
{
   /*
   ASN1T_ReturnResult.result.result(opentype) needs to be decoded
   using ASN1T_ReturnResult.result.opcode value as defined in Table 1.
   */

   if(pdu.u.returnResult->result.opcode.t != T_Code_local) {
      printf ("ROSE Operation's ReturnResult.result.result type is not known\n");
      return;
   }

   ASN1BERDecodeBuffer decodeBuffer2 (pdu.u.returnResult->result.result.data, 
      pdu.u.returnResult->result.result.numocts);
   int stat;

   switch(pdu.u.returnResult->result.opcode.u.local) {
      case 1 : {  /* alternateCall operation */
            ASN1T_AlternateCallResult alternateCallRes;
            ASN1C_AlternateCallResult alternateCallResC(decodeBuffer2, alternateCallRes);
            stat = alternateCallResC.Decode();
            if (stat == 0) {
               /* process the alternateCall operation in your application */
               if (tracearg) {
                  printf ("Decode of AlternateCallResult was successful\n");
                  alternateCallResC.Print ("AlternateCallResult");
               }
            }
            else {
               printf ("Decode of AlternateCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 2 : {  /* answerCall operation */
            ASN1T_AnswerCallResult answerCallRes;
            ASN1C_AnswerCallResult answerCallResC(decodeBuffer2, answerCallRes);
            stat = answerCallResC.Decode();
            if (stat == 0) {
               processAnswerCallRes(answerCallRes);
               if (tracearg) {
                  printf ("Decode of AnswerCallResult was successful\n");
                  answerCallResC.Print ("AnswerCallResult");
               }
            }
            else {
               printf ("Decode of AnswerCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 4 : {  /* clearCall operation */
            ASN1T_ClearCallResult clearCallRes;
            ASN1C_ClearCallResult clearCallResC(decodeBuffer2, clearCallRes);
            stat = clearCallResC.Decode();
            if (stat == 0) {
               /* process the clearCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ClearCallResult was successful\n");
                  clearCallResC.Print ("ClearCallResult");
               }
            }
            else {
               printf ("Decode of ClearCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 5 : {  /* clearConnection operation */
            ASN1T_ClearConnectionResult clearConnectionRes;
            ASN1C_ClearConnectionResult clearConnectionResC(decodeBuffer2, clearConnectionRes);
            stat = clearConnectionResC.Decode();
            if (stat == 0) {
               /* process the clearConnection operation in your application */
               if (tracearg) {
                  printf ("Decode of ClearConnectionResult was successful\n");
                  clearConnectionResC.Print ("ClearConnectionResult");
               }
            }
            else {
               printf ("Decode of ClearConnectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 6 : {  /* conferenceCall operation */
            ASN1T_ConferenceCallResult conferenceCallRes;
            ASN1C_ConferenceCallResult conferenceCallResC(decodeBuffer2, conferenceCallRes);
            stat = conferenceCallResC.Decode();
            if (stat == 0) {
               /* process the conferenceCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ConferenceCallResult was successful\n");
                  conferenceCallResC.Print ("ConferenceCallResult");
               }
            }
            else {
               printf ("Decode of ConferenceCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 7 : {  /* consultationCall operation */
            /* allocate on the heap since we'll need to reference part of this response
               later */
            ASN1T_ConsultationCallResult *pConsultationCallRes = new ASN1T_ConsultationCallResult();
            ASN1C_ConsultationCallResult *pConsultationCallResC = 
               new ASN1C_ConsultationCallResult(decodeBuffer2, *pConsultationCallRes);
            stat = pConsultationCallResC->Decode();
            if (stat == 0) {
               processConsCallRes(*pConsultationCallRes);
               if (tracearg) {
                  printf ("Decode of ConsultationCallResult was successful\n");
                  pConsultationCallResC->Print ("ConsultationCallResult");
               }
            }
            else {
               printf ("Decode of ConsultationCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 9 : {  /* holdCall operation */
            ASN1T_HoldCallResult holdCallRes;
            ASN1C_HoldCallResult holdCallResC(decodeBuffer2, holdCallRes);
            stat = holdCallResC.Decode();
            if (stat == 0) {
               /* process the holdCall operation in your application */
               if (tracearg) {
                  printf ("Decode of HoldCallResult was successful\n");
                  holdCallResC.Print ("HoldCallResult");
               }
            }
            else {
               printf ("Decode of HoldCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 10 : {  /* makeCall operation */
            /* allocate on the heap since we'll need to reference part of this response
               later */
            ASN1T_MakeCallResult *pMakeCallRes = new ASN1T_MakeCallResult();
            ASN1C_MakeCallResult *pMakeCallResC = new ASN1C_MakeCallResult(decodeBuffer2, *pMakeCallRes);
            stat = pMakeCallResC->Decode();
            if (stat == 0) {
               processMakeCallResult(*pMakeCallRes);
               if (tracearg) {
                  printf ("Decode of MakeCallResult was successful\n");
                  pMakeCallResC->Print ("MakeCallResult");
               }
            }
            else {
               printf ("Decode of MakeCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 11 : {  /* makePredictiveCall operation */
            ASN1T_MakePredictiveCallResult makePredictiveCallRes;
            ASN1C_MakePredictiveCallResult makePredictiveCallResC(decodeBuffer2, makePredictiveCallRes);
            stat = makePredictiveCallResC.Decode();
            if (stat == 0) {
               /* process the makePredictiveCall operation in your application */
               if (tracearg) {
                  printf ("Decode of MakePredictiveCallResult was successful\n");
                  makePredictiveCallResC.Print ("MakePredictiveCallResult");
               }
            }
            else {
               printf ("Decode of MakePredictiveCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 13 : {  /* reconnectCall operation */
            ASN1T_ReconnectCallResult reconnectCallRes;
            ASN1C_ReconnectCallResult reconnectCallResC(decodeBuffer2, reconnectCallRes);
            stat = reconnectCallResC.Decode();
            if (stat == 0) {
               /* process the reconnectCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ReconnectCallResult was successful\n");
                  reconnectCallResC.Print ("ReconnectCallResult");
               }
            }
            else {
               printf ("Decode of ReconnectCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 14 : {  /* retrieveCall operation */
            ASN1T_RetrieveCallResult retrieveCallRes;
            ASN1C_RetrieveCallResult retrieveCallResC(decodeBuffer2, retrieveCallRes);
            stat = retrieveCallResC.Decode();
            if (stat == 0) {
               /* process the retrieveCall operation in your application */
               if (tracearg) {
                  printf ("Decode of RetrieveCallResult was successful\n");
                  retrieveCallResC.Print ("RetrieveCallResult");
               }
            }
            else {
               printf ("Decode of RetrieveCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 16 : {  /* transferCall operation */
            ASN1T_TransferCallResult transferCallRes;
            ASN1C_TransferCallResult transferCallResC(decodeBuffer2, transferCallRes);
            stat = transferCallResC.Decode();
            if (stat == 0) {
               processTransferCallRes(transferCallRes);
               if (tracearg) {
                  printf ("Decode of TransferCallResult was successful\n");
                  transferCallResC.Print ("TransferCallResult");
               }
            }
            else {
               printf ("Decode of TransferCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 18 : {  /* parkCall operation */
            ASN1T_ParkCallResult parkCallRes;
            ASN1C_ParkCallResult parkCallResC(decodeBuffer2, parkCallRes);
            stat = parkCallResC.Decode();
            if (stat == 0) {
               /* process the parkCall operation in your application */
               if (tracearg) {
                  printf ("Decode of ParkCallResult was successful\n");
                  parkCallResC.Print ("ParkCallResult");
               }
            }
            else {
               printf ("Decode of ParkCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 20 : {  /* singleStepConf operation */
            ASN1T_SingleStepConfResult singleStepConfRes;
            ASN1C_SingleStepConfResult singleStepConfResC(decodeBuffer2, singleStepConfRes);
            stat = singleStepConfResC.Decode();
            if (stat == 0) {
               /* process the singleStepConf operation in your application */
               if (tracearg) {
                  printf ("Decode of SingleStepConfResult was successful\n");
                  singleStepConfResC.Print ("SingleStepConfResult");
               }
            }
            else {
               printf ("Decode of SingleStepConfResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 50 : {  /* singleStepTrans operation */
            ASN1T_SingleStepTransResult singleStepTransRes;
            ASN1C_SingleStepTransResult singleStepTransResC(decodeBuffer2, singleStepTransRes);
            stat = singleStepTransResC.Decode();
            if (stat == 0) {
               processSingleStepTransferRes(singleStepTransRes);

               if (tracearg) {
                  printf ("Decode of SingleStepTransResult was successful\n");
                  singleStepTransResC.Print ("SingleStepTransResult");
               }
            }
            else {
               printf ("Decode of SingleStepTransResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 51 : {  /* escape operation */
            ASN1T_EscapeResult escapeRes;
            ASN1C_EscapeResult escapeResC(decodeBuffer2, escapeRes);
            stat = escapeResC.Decode();
            if (stat == 0) {
               processEscapeServiceResult(escapeRes);
               if (tracearg) {
                  printf ("Decode of EscapeResult was successful\n");
                  escapeResC.Print ("EscapeResult");
               }
            }
            else {
               printf ("Decode of EscapeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 71 : {  /* monitorStart operation */
            ASN1T_MonitorStartResult monitorStartRes;
            ASN1C_MonitorStartResult monitorStartResC(decodeBuffer2, monitorStartRes);
            stat = monitorStartResC.Decode();
            if (stat == 0) {
               processMonitorStartRes(monitorStartRes);
               if (tracearg) {
                  printf ("Decode of MonitorStartResult was successful\n");
                  monitorStartResC.Print ("MonitorStartResult");
               }
            }
            else {
               printf ("Decode of MonitorStartResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 72 : {  /* changeMonitorFilter operation */
            ASN1T_ChangeMonitorFilterResult changeMonitorFilterRes;
            ASN1C_ChangeMonitorFilterResult changeMonitorFilterResC(decodeBuffer2, changeMonitorFilterRes);
            stat = changeMonitorFilterResC.Decode();
            if (stat == 0) {
               /* process the changeMonitorFilter operation in your application */
               if (tracearg) {
                  printf ("Decode of ChangeMonitorFilterResult was successful\n");
                  changeMonitorFilterResC.Print ("ChangeMonitorFilterResult");
               }
            }
            else {
               printf ("Decode of ChangeMonitorFilterResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 73 : {  /* monitorStop operation */
            ASN1T_MonitorStopResult monitorStopRes;
            ASN1C_MonitorStopResult monitorStopResC(decodeBuffer2, monitorStopRes);
            stat = monitorStopResC.Decode();
            if (stat == 0) {
               processMonitorStopRes(monitorStopRes);
               if (tracearg) {
                  printf ("Decode of MonitorStopResult was successful\n");
                  monitorStopResC.Print ("MonitorStopResult");
               }
            }
            else {
               printf ("Decode of MonitorStopResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 74 : {  /* snapshotDevice operation */
            ASN1T_SnapshotDeviceResult snapshotDeviceRes;
            ASN1C_SnapshotDeviceResult snapshotDeviceResC(decodeBuffer2, snapshotDeviceRes);
            stat = snapshotDeviceResC.Decode();
            if (stat == 0) {
               /* process the snapshotDevice operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotDeviceResult was successful\n");
                  snapshotDeviceResC.Print ("SnapshotDeviceResult");
               }
            }
            else {
               printf ("Decode of SnapshotDeviceResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 75 : {  /* snapshotCall operation */
            ASN1T_SnapshotCallResult snapshotCallRes;
            ASN1C_SnapshotCallResult snapshotCallResC(decodeBuffer2, snapshotCallRes);
            stat = snapshotCallResC.Decode();
            if (stat == 0) {
               /* process the snapshotCall operation in your application */
               if (tracearg) {
                  printf ("Decode of SnapshotCallResult was successful\n");
                  snapshotCallResC.Print ("SnapshotCallResult");
               }
            }
            else {
               printf ("Decode of SnapshotCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 110 : {  /* startDataPath operation */
            ASN1T_StartDataPathResult startDataPathRes;
            ASN1C_StartDataPathResult startDataPathResC(decodeBuffer2, startDataPathRes);
            stat = startDataPathResC.Decode();
            if (stat == 0) {
               /* process the startDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of StartDataPathResult was successful\n");
                  startDataPathResC.Print ("StartDataPathResult");
               }
            }
            else {
               printf ("Decode of StartDataPathResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 111 : {  /* stopDataPath operation */
            ASN1T_StopDataPathResult stopDataPathRes;
            ASN1C_StopDataPathResult stopDataPathResC(decodeBuffer2, stopDataPathRes);
            stat = stopDataPathResC.Decode();
            if (stat == 0) {
               /* process the stopDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of StopDataPathResult was successful\n");
                  stopDataPathResC.Print ("StopDataPathResult");
               }
            }
            else {
               printf ("Decode of StopDataPathResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 112 : {  /* sendData operation */
            ASN1T_SendDataResult sendDataRes;
            ASN1C_SendDataResult sendDataResC(decodeBuffer2, sendDataRes);
            stat = sendDataResC.Decode();
            if (stat == 0) {
               /* process the sendData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendDataResult was successful\n");
                  sendDataResC.Print ("SendDataResult");
               }
            }
            else {
               printf ("Decode of SendDataResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 113 : {  /* sendMulticastData operation */
            ASN1T_SendMulticastDataResult sendMulticastDataRes;
            ASN1C_SendMulticastDataResult sendMulticastDataResC(decodeBuffer2, sendMulticastDataRes);
            stat = sendMulticastDataResC.Decode();
            if (stat == 0) {
               /* process the sendMulticastData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendMulticastDataResult was successful\n");
                  sendMulticastDataResC.Print ("SendMulticastDataResult");
               }
            }
            else {
               printf ("Decode of SendMulticastDataResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 114 : {  /* sendBroadcastData operation */
            ASN1T_SendBroadcastDataResult sendBroadcastDataRes;
            ASN1C_SendBroadcastDataResult sendBroadcastDataResC(decodeBuffer2, sendBroadcastDataRes);
            stat = sendBroadcastDataResC.Decode();
            if (stat == 0) {
               /* process the sendBroadcastData operation in your application */
               if (tracearg) {
                  printf ("Decode of SendBroadcastDataResult was successful\n");
                  sendBroadcastDataResC.Print ("SendBroadcastDataResult");
               }
            }
            else {
               printf ("Decode of SendBroadcastDataResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 115 : {  /* suspendDataPath operation */
            ASN1T_SuspendDataPathResult suspendDataPathRes;
            ASN1C_SuspendDataPathResult suspendDataPathResC(decodeBuffer2, suspendDataPathRes);
            stat = suspendDataPathResC.Decode();
            if (stat == 0) {
               /* process the suspendDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendDataPathResult was successful\n");
                  suspendDataPathResC.Print ("SuspendDataPathResult");
               }
            }
            else {
               printf ("Decode of SuspendDataPathResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 116 : {  /* dataPathSuspended operation */
            ASN1T_DataPathSuspendedResult dataPathSuspendedRes;
            ASN1C_DataPathSuspendedResult dataPathSuspendedResC(decodeBuffer2, dataPathSuspendedRes);
            stat = dataPathSuspendedResC.Decode();
            if (stat == 0) {
               /* process the dataPathSuspended operation in your application */
               if (tracearg) {
                  printf ("Decode of DataPathSuspendedResult was successful\n");
                  dataPathSuspendedResC.Print ("DataPathSuspendedResult");
               }
            }
            else {
               printf ("Decode of DataPathSuspendedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 117 : {  /* resumeDataPath operation */
            ASN1T_ResumeDataPathResult resumeDataPathRes;
            ASN1C_ResumeDataPathResult resumeDataPathResC(decodeBuffer2, resumeDataPathRes);
            stat = resumeDataPathResC.Decode();
            if (stat == 0) {
               /* process the resumeDataPath operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeDataPathResult was successful\n");
                  resumeDataPathResC.Print ("ResumeDataPathResult");
               }
            }
            else {
               printf ("Decode of ResumeDataPathResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 118 : {  /* dataPathResumed operation */
            ASN1T_DataPathResumedResult dataPathResumedRes;
            ASN1C_DataPathResumedResult dataPathResumedResC(decodeBuffer2, dataPathResumedRes);
            stat = dataPathResumedResC.Decode();
            if (stat == 0) {
               /* process the dataPathResumed operation in your application */
               if (tracearg) {
                  printf ("Decode of DataPathResumedResult was successful\n");
                  dataPathResumedResC.Print ("DataPathResumedResult");
               }
            }
            else {
               printf ("Decode of DataPathResumedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 119 : {  /* fastData operation */
            ASN1T_FastDataResult fastDataRes;
            ASN1C_FastDataResult fastDataResC(decodeBuffer2, fastDataRes);
            stat = fastDataResC.Decode();
            if (stat == 0) {
               /* process the fastData operation in your application */
               if (tracearg) {
                  printf ("Decode of FastDataResult was successful\n");
                  fastDataResC.Print ("FastDataResult");
               }
            }
            else {
               printf ("Decode of FastDataResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 201 : {  /* getLogicalDeviceInformation operation */
            ASN1T_GetLogicalDeviceInformationResult getLogicalDeviceInformationRes;
            ASN1C_GetLogicalDeviceInformationResult getLogicalDeviceInformationResC(decodeBuffer2, getLogicalDeviceInformationRes);
            stat = getLogicalDeviceInformationResC.Decode();
            if (stat == 0) {
               /* process the getLogicalDeviceInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLogicalDeviceInformationResult was successful\n");
                  getLogicalDeviceInformationResC.Print ("GetLogicalDeviceInformationResult");
               }
            }
            else {
               printf ("Decode of GetLogicalDeviceInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 202 : {  /* getPhysicalDeviceInformation operation */
            ASN1T_GetPhysicalDeviceInformationResult getPhysicalDeviceInformationRes;
            ASN1C_GetPhysicalDeviceInformationResult getPhysicalDeviceInformationResC(decodeBuffer2, getPhysicalDeviceInformationRes);
            stat = getPhysicalDeviceInformationResC.Decode();
            if (stat == 0) {
               /* process the getPhysicalDeviceInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetPhysicalDeviceInformationResult was successful\n");
                  getPhysicalDeviceInformationResC.Print ("GetPhysicalDeviceInformationResult");
               }
            }
            else {
               printf ("Decode of GetPhysicalDeviceInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 203 : {  /* getSwitchingFunctionCapabilities operation */
            ASN1T_GetSwitchingFunctionCapsResult getSwitchingFunctionCapabilitiesRes;
            ASN1C_GetSwitchingFunctionCapsResult getSwitchingFunctionCapabilitiesResC(decodeBuffer2, getSwitchingFunctionCapabilitiesRes);
            stat = getSwitchingFunctionCapabilitiesResC.Decode();
            if (stat == 0) {
               /* process the getSwitchingFunctionCapabilities operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSwitchingFunctionCapsResult was successful\n");
                  getSwitchingFunctionCapabilitiesResC.Print ("GetSwitchingFunctionCapsResult");
               }
            }
            else {
               printf ("Decode of GetSwitchingFunctionCapsResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 204 : {  /* getSwitchingFunctionDevices operation */
            ASN1T_GetSwitchingFunctionDevicesResult getSwitchingFunctionDevicesRes;
            ASN1C_GetSwitchingFunctionDevicesResult getSwitchingFunctionDevicesResC(decodeBuffer2, getSwitchingFunctionDevicesRes);
            stat = getSwitchingFunctionDevicesResC.Decode();
            if (stat == 0) {
               /* process the getSwitchingFunctionDevices operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSwitchingFunctionDevicesResult was successful\n");
                  getSwitchingFunctionDevicesResC.Print ("GetSwitchingFunctionDevicesResult");
               }
            }
            else {
               printf ("Decode of GetSwitchingFunctionDevicesResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 206 : {  /* changeSysStatFilter operation */
            ASN1T_ChangeSysStatFilterRes changeSysStatFilterRes;
            ASN1C_ChangeSysStatFilterRes changeSysStatFilterResC(decodeBuffer2, changeSysStatFilterRes);
            stat = changeSysStatFilterResC.Decode();
            if (stat == 0) {
               /* process the changeSysStatFilter operation in your application */
               if (tracearg) {
                  printf ("Decode of ChangeSysStatFilterRes was successful\n");
                  changeSysStatFilterResC.Print ("ChangeSysStatFilterRes");
               }
            }
            else {
               printf ("Decode of ChangeSysStatFilterRes failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 207 : {  /* systemRegister operation */
            ASN1T_SystemRegisterResult systemRegisterRes;
            ASN1C_SystemRegisterResult systemRegisterResC(decodeBuffer2, systemRegisterRes);
            stat = systemRegisterResC.Decode();
            if (stat == 0) {
               /* process the systemRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemRegisterResult was successful\n");
                  systemRegisterResC.Print ("SystemRegisterResult");
               }
            }
            else {
               printf ("Decode of SystemRegisterResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 209 : {  /* systemRegisterCancel operation */
            ASN1T_SystemRegisterCancelResult systemRegisterCancelRes;
            ASN1C_SystemRegisterCancelResult systemRegisterCancelResC(decodeBuffer2, systemRegisterCancelRes);
            stat = systemRegisterCancelResC.Decode();
            if (stat == 0) {
               /* process the systemRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemRegisterCancelResult was successful\n");
                  systemRegisterCancelResC.Print ("SystemRegisterCancelResult");
               }
            }
            else {
               printf ("Decode of SystemRegisterCancelResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 210 : {  /* requestSysStat operation */
            ASN1T_RequestSysStatRes requestSysStatRes;
            ASN1C_RequestSysStatRes requestSysStatResC(decodeBuffer2, requestSysStatRes);
            stat = requestSysStatResC.Decode();
            if (stat == 0) {
               /* process the requestSysStat operation in your application */
               if (tracearg) {
                  printf ("Decode of RequestSysStatRes was successful\n");
                  requestSysStatResC.Print ("RequestSysStatRes");
               }
            }
            else {
               printf ("Decode of RequestSysStatRes failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 211 : {  /* systemStatus operation */
            ASN1T_SystemStatusRes systemStatusRes;
            ASN1C_SystemStatusRes systemStatusResC(decodeBuffer2, systemStatusRes);
            stat = systemStatusResC.Decode();
            if (stat == 0) {
               /* process the systemStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SystemStatusRes was successful\n");
                  systemStatusResC.Print ("SystemStatusRes");
               }
            }
            else {
               printf ("Decode of SystemStatusRes failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 212 : {  /* swFunctionCapsChanged operation */
            ASN1T_SwFunctionCapsChangedRes swFunctionCapsChangedRes;
            ASN1C_SwFunctionCapsChangedRes swFunctionCapsChangedResC(decodeBuffer2, swFunctionCapsChangedRes);
            stat = swFunctionCapsChangedResC.Decode();
            if (stat == 0) {
               /* process the swFunctionCapsChanged operation in your application */
               if (tracearg) {
                  printf ("Decode of SwFunctionCapsChangedRes was successful\n");
                  swFunctionCapsChangedResC.Print ("SwFunctionCapsChangedRes");
               }
            }
            else {
               printf ("Decode of SwFunctionCapsChangedRes failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 213 : {  /* swFunctionDevicesChanged operation */
            ASN1T_SwFunctionDevicesChangedRes swFunctionDevicesChangedRes;
            ASN1C_SwFunctionDevicesChangedRes swFunctionDevicesChangedResC(decodeBuffer2, swFunctionDevicesChangedRes);
            stat = swFunctionDevicesChangedResC.Decode();
            if (stat == 0) {
               /* process the swFunctionDevicesChanged operation in your application */
               if (tracearg) {
                  printf ("Decode of SwFunctionDevicesChangedRes was successful\n");
                  swFunctionDevicesChangedResC.Print ("SwFunctionDevicesChangedRes");
               }
            }
            else {
               printf ("Decode of SwFunctionDevicesChangedRes failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 214 : {  /* acceptCall operation */
            ASN1T_AcceptCallResult acceptCallRes;
            ASN1C_AcceptCallResult acceptCallResC(decodeBuffer2, acceptCallRes);
            stat = acceptCallResC.Decode();
            if (stat == 0) {
               /* process the acceptCall operation in your application */
               if (tracearg) {
                  printf ("Decode of AcceptCallResult was successful\n");
                  acceptCallResC.Print ("AcceptCallResult");
               }
            }
            else {
               printf ("Decode of AcceptCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 215 : {  /* callBack operation */
            ASN1T_CallBackResult callBackRes;
            ASN1C_CallBackResult callBackResC(decodeBuffer2, callBackRes);
            stat = callBackResC.Decode();
            if (stat == 0) {
               /* process the callBack operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackResult was successful\n");
                  callBackResC.Print ("CallBackResult");
               }
            }
            else {
               printf ("Decode of CallBackResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 216 : {  /* callBackMessage operation */
            ASN1T_CallBackMessageResult callBackMessageRes;
            ASN1C_CallBackMessageResult callBackMessageResC(decodeBuffer2, callBackMessageRes);
            stat = callBackMessageResC.Decode();
            if (stat == 0) {
               /* process the callBackMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackMessageResult was successful\n");
                  callBackMessageResC.Print ("CallBackMessageResult");
               }
            }
            else {
               printf ("Decode of CallBackMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 217 : {  /* camponCall operation */
            ASN1T_CamponCallResult camponCallRes;
            ASN1C_CamponCallResult camponCallResC(decodeBuffer2, camponCallRes);
            stat = camponCallResC.Decode();
            if (stat == 0) {
               /* process the camponCall operation in your application */
               if (tracearg) {
                  printf ("Decode of CamponCallResult was successful\n");
                  camponCallResC.Print ("CamponCallResult");
               }
            }
            else {
               printf ("Decode of CamponCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;


      case 218 : {  /* deflectCall operation */
            ASN1T_DeflectCallResult deflectCallRes;
            ASN1C_DeflectCallResult deflectCallResC(decodeBuffer2, deflectCallRes);
            stat = deflectCallResC.Decode();
            if (stat == 0) {
               /* process the deflectCall operation in your application */
               if (tracearg) {
                  printf ("Decode of DeflectCallResult was successful\n");
                  deflectCallResC.Print ("DeflectCallResult");
               }
            }
            else {
               printf ("Decode of DeflectCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 219 : {  /* dialDigits operation */
            ASN1T_DialDigitsResult dialDigitsRes;
            ASN1C_DialDigitsResult dialDigitsResC(decodeBuffer2, dialDigitsRes);
            stat = dialDigitsResC.Decode();
            if (stat == 0) {
               /* process the dialDigits operation in your application */
               if (tracearg) {
                  printf ("Decode of DialDigitsResult was successful\n");
                  dialDigitsResC.Print ("DialDigitsResult");
               }
            }
            else {
               printf ("Decode of DialDigitsResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 220 : {  /* directedPickupCall operation */
            ASN1T_DirectedPickupCallResult directedPickupCallRes;
            ASN1C_DirectedPickupCallResult directedPickupCallResC(decodeBuffer2, directedPickupCallRes);
            stat = directedPickupCallResC.Decode();
            if (stat == 0) {
               /* process the directedPickupCall operation in your application */
               if (tracearg) {
                  printf ("Decode of DirectedPickupCallResult was successful\n");
                  directedPickupCallResC.Print ("DirectedPickupCallResult");
               }
            }
            else {
               printf ("Decode of DirectedPickupCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 221 : {  /* groupPickupCall operation */
            ASN1T_GroupPickupCallResult groupPickupCallRes;
            ASN1C_GroupPickupCallResult groupPickupCallResC(decodeBuffer2, groupPickupCallRes);
            stat = groupPickupCallResC.Decode();
            if (stat == 0) {
               /* process the groupPickupCall operation in your application */
               if (tracearg) {
                  printf ("Decode of GroupPickupCallResult was successful\n");
                  groupPickupCallResC.Print ("GroupPickupCallResult");
               }
            }
            else {
               printf ("Decode of GroupPickupCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 222 : {  /* intrudeCall operation */
            ASN1T_IntrudeCallResult intrudeCallRes;
            ASN1C_IntrudeCallResult intrudeCallResC(decodeBuffer2, intrudeCallRes);
            stat = intrudeCallResC.Decode();
            if (stat == 0) {
               /* process the intrudeCall operation in your application */
               if (tracearg) {
                  printf ("Decode of IntrudeCallResult was successful\n");
                  intrudeCallResC.Print ("IntrudeCallResult");
               }
            }
            else {
               printf ("Decode of IntrudeCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 223 : {  /* joinCall operation */
            ASN1T_JoinCallResult joinCallRes;
            ASN1C_JoinCallResult joinCallResC(decodeBuffer2, joinCallRes);
            stat = joinCallResC.Decode();
            if (stat == 0) {
               /* process the joinCall operation in your application */
               if (tracearg) {
                  printf ("Decode of JoinCallResult was successful\n");
                  joinCallResC.Print ("JoinCallResult");
               }
            }
            else {
               printf ("Decode of JoinCallResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 230 : {  /* associateData operation */
            ASN1T_AssociateDataResult associateDataRes;
            ASN1C_AssociateDataResult associateDataResC(decodeBuffer2, associateDataRes);
            stat = associateDataResC.Decode();
            if (stat == 0) {
               /* process the associateData operation in your application */
               if (tracearg) {
                  printf ("Decode of AssociateDataResult was successful\n");
                  associateDataResC.Print ("AssociateDataResult");
               }
            }
            else {
               printf ("Decode of AssociateDataResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 231 : {  /* cancelTelephonyTones operation */
            ASN1T_CancelTelephonyTonesResult cancelTelephonyTonesRes;
            ASN1C_CancelTelephonyTonesResult cancelTelephonyTonesResC(decodeBuffer2, cancelTelephonyTonesRes);
            stat = cancelTelephonyTonesResC.Decode();
            if (stat == 0) {
               /* process the cancelTelephonyTones operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelTelephonyTonesResult was successful\n");
                  cancelTelephonyTonesResC.Print ("CancelTelephonyTonesResult");
               }
            }
            else {
               printf ("Decode of CancelTelephonyTonesResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 232 : {  /* generateDigits operation */
            ASN1T_GenerateDigitsResult generateDigitsRes;
            ASN1C_GenerateDigitsResult generateDigitsResC(decodeBuffer2, generateDigitsRes);
            stat = generateDigitsResC.Decode();
            if (stat == 0) {
               /* process the generateDigits operation in your application */
               if (tracearg) {
                  printf ("Decode of GenerateDigitsResult was successful\n");
                  generateDigitsResC.Print ("GenerateDigitsResult");
               }
            }
            else {
               printf ("Decode of GenerateDigitsResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 233 : {  /* generateTelephonyTones operation */
            ASN1T_GenerateTelephonyTonesResult generateTelephonyTonesRes;
            ASN1C_GenerateTelephonyTonesResult generateTelephonyTonesResC(decodeBuffer2, generateTelephonyTonesRes);
            stat = generateTelephonyTonesResC.Decode();
            if (stat == 0) {
               /* process the generateTelephonyTones operation in your application */
               if (tracearg) {
                  printf ("Decode of GenerateTelephonyTonesResult was successful\n");
                  generateTelephonyTonesResC.Print ("GenerateTelephonyTonesResult");
               }
            }
            else {
               printf ("Decode of GenerateTelephonyTonesResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 234 : {  /* sendUserInfo operation */
            ASN1T_SendUserInfoResult sendUserInfoRes;
            ASN1C_SendUserInfoResult sendUserInfoResC(decodeBuffer2, sendUserInfoRes);
            stat = sendUserInfoResC.Decode();
            if (stat == 0) {
               /* process the sendUserInfo operation in your application */
               if (tracearg) {
                  printf ("Decode of SendUserInfoResult was successful\n");
                  sendUserInfoResC.Print ("SendUserInfoResult");
               }
            }
            else {
               printf ("Decode of SendUserInfoResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 244 : {  /* attachMediaService operation */
            ASN1T_AttachMediaServiceResult attachMediaServiceRes;
            ASN1C_AttachMediaServiceResult attachMediaServiceResC(decodeBuffer2, attachMediaServiceRes);
            stat = attachMediaServiceResC.Decode();
            if (stat == 0) {
               /* process the attachMediaService operation in your application */
               if (tracearg) {
                  printf ("Decode of AttachMediaServiceResult was successful\n");
                  attachMediaServiceResC.Print ("AttachMediaServiceResult");
               }
            }
            else {
               printf ("Decode of AttachMediaServiceResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 245 : {  /* detachMediaService operation */
            ASN1T_DetachMediaServiceResult detachMediaServiceRes;
            ASN1C_DetachMediaServiceResult detachMediaServiceResC(decodeBuffer2, detachMediaServiceRes);
            stat = detachMediaServiceResC.Decode();
            if (stat == 0) {
               /* process the detachMediaService operation in your application */
               if (tracearg) {
                  printf ("Decode of DetachMediaServiceResult was successful\n");
                  detachMediaServiceResC.Print ("DetachMediaServiceResult");
               }
            }
            else {
               printf ("Decode of DetachMediaServiceResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 248 : {  /* routeRegister operation */
            ASN1T_RouteRegisterResult routeRegisterRes;
            ASN1C_RouteRegisterResult routeRegisterResC(decodeBuffer2, routeRegisterRes);
            stat = routeRegisterResC.Decode();
            if (stat == 0) {
               /* process the routeRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRegisterResult was successful\n");
                  routeRegisterResC.Print ("RouteRegisterResult");
               }
            }
            else {
               printf ("Decode of RouteRegisterResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 250 : {  /* routeRegisterCancel operation */
            ASN1T_RouteRegisterCancelResult routeRegisterCancelRes;
            ASN1C_RouteRegisterCancelResult routeRegisterCancelResC(decodeBuffer2, routeRegisterCancelRes);
            stat = routeRegisterCancelResC.Decode();
            if (stat == 0) {
               /* process the routeRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of RouteRegisterCancelResult was successful\n");
                  routeRegisterCancelResC.Print ("RouteRegisterCancelResult");
               }
            }
            else {
               printf ("Decode of RouteRegisterCancelResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 260 : {  /* buttonPress operation */
            ASN1T_ButtonPressResult buttonPressRes;
            ASN1C_ButtonPressResult buttonPressResC(decodeBuffer2, buttonPressRes);
            stat = buttonPressResC.Decode();
            if (stat == 0) {
               /* process the buttonPress operation in your application */
               if (tracearg) {
                  printf ("Decode of ButtonPressResult was successful\n");
                  buttonPressResC.Print ("ButtonPressResult");
               }
            }
            else {
               printf ("Decode of ButtonPressResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 261 : {  /* getAuditoryApparatusInformation operation */
            ASN1T_GetAuditoryApparatusInformationResult getAuditoryApparatusInformationRes;
            ASN1C_GetAuditoryApparatusInformationResult getAuditoryApparatusInformationResC(decodeBuffer2, getAuditoryApparatusInformationRes);
            stat = getAuditoryApparatusInformationResC.Decode();
            if (stat == 0) {
               /* process the getAuditoryApparatusInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAuditoryApparatusInformationResult was successful\n");
                  getAuditoryApparatusInformationResC.Print ("GetAuditoryApparatusInformationResult");
               }
            }
            else {
               printf ("Decode of GetAuditoryApparatusInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 262 : {  /* getButtonInformation operation */
            ASN1T_GetButtonInformationResult getButtonInformationRes;
            ASN1C_GetButtonInformationResult getButtonInformationResC(decodeBuffer2, getButtonInformationRes);
            stat = getButtonInformationResC.Decode();
            if (stat == 0) {
               /* process the getButtonInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetButtonInformationResult was successful\n");
                  getButtonInformationResC.Print ("GetButtonInformationResult");
               }
            }
            else {
               printf ("Decode of GetButtonInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 263 : {  /* getDisplay operation */
            ASN1T_GetDisplayResult getDisplayRes;
            ASN1C_GetDisplayResult getDisplayResC(decodeBuffer2, getDisplayRes);
            stat = getDisplayResC.Decode();
            if (stat == 0) {
               /* process the getDisplay operation in your application */
               if (tracearg) {
                  printf ("Decode of GetDisplayResult was successful\n");
                  getDisplayResC.Print ("GetDisplayResult");
               }
            }
            else {
               printf ("Decode of GetDisplayResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 264 : {  /* getHookswitchStatus operation */
            ASN1T_GetHookswitchStatusResult getHookswitchStatusRes;
            ASN1C_GetHookswitchStatusResult getHookswitchStatusResC(decodeBuffer2, getHookswitchStatusRes);
            stat = getHookswitchStatusResC.Decode();
            if (stat == 0) {
               /* process the getHookswitchStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetHookswitchStatusResult was successful\n");
                  getHookswitchStatusResC.Print ("GetHookswitchStatusResult");
               }
            }
            else {
               printf ("Decode of GetHookswitchStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 265 : {  /* getLampInformation operation */
            ASN1T_GetLampInformationResult getLampInformationRes;
            ASN1C_GetLampInformationResult getLampInformationResC(decodeBuffer2, getLampInformationRes);
            stat = getLampInformationResC.Decode();
            if (stat == 0) {
               /* process the getLampInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLampInformationResult was successful\n");
                  getLampInformationResC.Print ("GetLampInformationResult");
               }
            }
            else {
               printf ("Decode of GetLampInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 266 : {  /* getLampMode operation */
            ASN1T_GetLampModeResult getLampModeRes;
            ASN1C_GetLampModeResult getLampModeResC(decodeBuffer2, getLampModeRes);
            stat = getLampModeResC.Decode();
            if (stat == 0) {
               /* process the getLampMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLampModeResult was successful\n");
                  getLampModeResC.Print ("GetLampModeResult");
               }
            }
            else {
               printf ("Decode of GetLampModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 267 : {  /* getMessageWaitingIndicator operation */
            ASN1T_GetMessageWaitingIndicatorResult getMessageWaitingIndicatorRes;
            ASN1C_GetMessageWaitingIndicatorResult getMessageWaitingIndicatorResC(decodeBuffer2, getMessageWaitingIndicatorRes);
            stat = getMessageWaitingIndicatorResC.Decode();
            if (stat == 0) {
               /* process the getMessageWaitingIndicator operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMessageWaitingIndicatorResult was successful\n");
                  getMessageWaitingIndicatorResC.Print ("GetMessageWaitingIndicatorResult");
               }
            }
            else {
               printf ("Decode of GetMessageWaitingIndicatorResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 268 : {  /* getMicrophoneGain operation */
            ASN1T_GetMicrophoneGainResult getMicrophoneGainRes;
            ASN1C_GetMicrophoneGainResult getMicrophoneGainResC(decodeBuffer2, getMicrophoneGainRes);
            stat = getMicrophoneGainResC.Decode();
            if (stat == 0) {
               /* process the getMicrophoneGain operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMicrophoneGainResult was successful\n");
                  getMicrophoneGainResC.Print ("GetMicrophoneGainResult");
               }
            }
            else {
               printf ("Decode of GetMicrophoneGainResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 269 : {  /* getMicrophoneMute operation */
            ASN1T_GetMicrophoneMuteResult getMicrophoneMuteRes;
            ASN1C_GetMicrophoneMuteResult getMicrophoneMuteResC(decodeBuffer2, getMicrophoneMuteRes);
            stat = getMicrophoneMuteResC.Decode();
            if (stat == 0) {
               /* process the getMicrophoneMute operation in your application */
               if (tracearg) {
                  printf ("Decode of GetMicrophoneMuteResult was successful\n");
                  getMicrophoneMuteResC.Print ("GetMicrophoneMuteResult");
               }
            }
            else {
               printf ("Decode of GetMicrophoneMuteResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 270 : {  /* getRingerStatus operation */
            ASN1T_GetRingerStatusResult getRingerStatusRes;
            ASN1C_GetRingerStatusResult getRingerStatusResC(decodeBuffer2, getRingerStatusRes);
            stat = getRingerStatusResC.Decode();
            if (stat == 0) {
               /* process the getRingerStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetRingerStatusResult was successful\n");
                  getRingerStatusResC.Print ("GetRingerStatusResult");
               }
            }
            else {
               printf ("Decode of GetRingerStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 271 : {  /* getSpeakerMute operation */
            ASN1T_GetSpeakerMuteResult getSpeakerMuteRes;
            ASN1C_GetSpeakerMuteResult getSpeakerMuteResC(decodeBuffer2, getSpeakerMuteRes);
            stat = getSpeakerMuteResC.Decode();
            if (stat == 0) {
               /* process the getSpeakerMute operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSpeakerMuteResult was successful\n");
                  getSpeakerMuteResC.Print ("GetSpeakerMuteResult");
               }
            }
            else {
               printf ("Decode of GetSpeakerMuteResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 272 : {  /* getSpeakerVolume operation */
            ASN1T_GetSpeakerVolumeResult getSpeakerVolumeRes;
            ASN1C_GetSpeakerVolumeResult getSpeakerVolumeResC(decodeBuffer2, getSpeakerVolumeRes);
            stat = getSpeakerVolumeResC.Decode();
            if (stat == 0) {
               /* process the getSpeakerVolume operation in your application */
               if (tracearg) {
                  printf ("Decode of GetSpeakerVolumeResult was successful\n");
                  getSpeakerVolumeResC.Print ("GetSpeakerVolumeResult");
               }
            }
            else {
               printf ("Decode of GetSpeakerVolumeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 273 : {  /* setButtonInformation operation */
            ASN1T_SetButtonInformationResult setButtonInformationRes;
            ASN1C_SetButtonInformationResult setButtonInformationResC(decodeBuffer2, setButtonInformationRes);
            stat = setButtonInformationResC.Decode();
            if (stat == 0) {
               /* process the setButtonInformation operation in your application */
               if (tracearg) {
                  printf ("Decode of SetButtonInformationResult was successful\n");
                  setButtonInformationResC.Print ("SetButtonInformationResult");
               }
            }
            else {
               printf ("Decode of SetButtonInformationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 274 : {  /* setDisplay operation */
            ASN1T_SetDisplayResult setDisplayRes;
            ASN1C_SetDisplayResult setDisplayResC(decodeBuffer2, setDisplayRes);
            stat = setDisplayResC.Decode();
            if (stat == 0) {
               /* process the setDisplay operation in your application */
               if (tracearg) {
                  printf ("Decode of SetDisplayResult was successful\n");
                  setDisplayResC.Print ("SetDisplayResult");
               }
            }
            else {
               printf ("Decode of SetDisplayResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 275 : {  /* setHookswitchStatus operation */
            ASN1T_SetHookswitchStatusResult setHookswitchStatusRes;
            ASN1C_SetHookswitchStatusResult setHookswitchStatusResC(decodeBuffer2, setHookswitchStatusRes);
            stat = setHookswitchStatusResC.Decode();
            if (stat == 0) {
               /* process the setHookswitchStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetHookswitchStatusResult was successful\n");
                  setHookswitchStatusResC.Print ("SetHookswitchStatusResult");
               }
            }
            else {
               printf ("Decode of SetHookswitchStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 276 : {  /* setLampMode operation */
            ASN1T_SetLampModeResult setLampModeRes;
            ASN1C_SetLampModeResult setLampModeResC(decodeBuffer2, setLampModeRes);
            stat = setLampModeResC.Decode();
            if (stat == 0) {
               /* process the setLampMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetLampModeResult was successful\n");
                  setLampModeResC.Print ("SetLampModeResult");
               }
            }
            else {
               printf ("Decode of SetLampModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 277 : {  /* setMessageWaitingIndicator operation */
            ASN1T_SetMessageWaitingIndicatorResult setMessageWaitingIndicatorRes;
            ASN1C_SetMessageWaitingIndicatorResult setMessageWaitingIndicatorResC(decodeBuffer2, setMessageWaitingIndicatorRes);
            stat = setMessageWaitingIndicatorResC.Decode();
            if (stat == 0) {
               /* process the setMessageWaitingIndicator operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMessageWaitingIndicatorResult was successful\n");
                  setMessageWaitingIndicatorResC.Print ("SetMessageWaitingIndicatorResult");
               }
            }
            else {
               printf ("Decode of SetMessageWaitingIndicatorResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 278 : {  /* setMicrophoneGain operation */
            ASN1T_SetMicrophoneGainResult setMicrophoneGainRes;
            ASN1C_SetMicrophoneGainResult setMicrophoneGainResC(decodeBuffer2, setMicrophoneGainRes);
            stat = setMicrophoneGainResC.Decode();
            if (stat == 0) {
               /* process the setMicrophoneGain operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMicrophoneGainResult was successful\n");
                  setMicrophoneGainResC.Print ("SetMicrophoneGainResult");
               }
            }
            else {
               printf ("Decode of SetMicrophoneGainResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 279 : {  /* setMicrophoneMute operation */
            ASN1T_SetMicrophoneMuteResult setMicrophoneMuteRes;
            ASN1C_SetMicrophoneMuteResult setMicrophoneMuteResC(decodeBuffer2, setMicrophoneMuteRes);
            stat = setMicrophoneMuteResC.Decode();
            if (stat == 0) {
               /* process the setMicrophoneMute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetMicrophoneMuteResult was successful\n");
                  setMicrophoneMuteResC.Print ("SetMicrophoneMuteResult");
               }
            }
            else {
               printf ("Decode of SetMicrophoneMuteResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 280 : {  /* setRingerStatus operation */
            ASN1T_SetRingerStatusResult setRingerStatusRes;
            ASN1C_SetRingerStatusResult setRingerStatusResC(decodeBuffer2, setRingerStatusRes);
            stat = setRingerStatusResC.Decode();
            if (stat == 0) {
               /* process the setRingerStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetRingerStatusResult was successful\n");
                  setRingerStatusResC.Print ("SetRingerStatusResult");
               }
            }
            else {
               printf ("Decode of SetRingerStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 281 : {  /* setSpeakerMute operation */
            ASN1T_SetSpeakerMuteResult setSpeakerMuteRes;
            ASN1C_SetSpeakerMuteResult setSpeakerMuteResC(decodeBuffer2, setSpeakerMuteRes);
            stat = setSpeakerMuteResC.Decode();
            if (stat == 0) {
               /* process the setSpeakerMute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetSpeakerMuteResult was successful\n");
                  setSpeakerMuteResC.Print ("SetSpeakerMuteResult");
               }
            }
            else {
               printf ("Decode of SetSpeakerMuteResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 282 : {  /* setSpeakerVolume operation */
            ASN1T_SetSpeakerVolumeResult setSpeakerVolumeRes;
            ASN1C_SetSpeakerVolumeResult setSpeakerVolumeResC(decodeBuffer2, setSpeakerVolumeRes);
            stat = setSpeakerVolumeResC.Decode();
            if (stat == 0) {
               /* process the setSpeakerVolume operation in your application */
               if (tracearg) {
                  printf ("Decode of SetSpeakerVolumeResult was successful\n");
                  setSpeakerVolumeResC.Print ("SetSpeakerVolumeResult");
               }
            }
            else {
               printf ("Decode of SetSpeakerVolumeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 300 : {  /* callBackNonCallRelated operation */
            ASN1T_CallBackNonCallRelatedResult callBackNonCallRelatedRes;
            ASN1C_CallBackNonCallRelatedResult callBackNonCallRelatedResC(decodeBuffer2, callBackNonCallRelatedRes);
            stat = callBackNonCallRelatedResC.Decode();
            if (stat == 0) {
               /* process the callBackNonCallRelated operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackNonCallRelatedResult was successful\n");
                  callBackNonCallRelatedResC.Print ("CallBackNonCallRelatedResult");
               }
            }
            else {
               printf ("Decode of CallBackNonCallRelatedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 301 : {  /* callBackMessageNonCallRelated operation */
            ASN1T_CallBackMessageNonCallRelatedResult callBackMessageNonCallRelatedRes;
            ASN1C_CallBackMessageNonCallRelatedResult callBackMessageNonCallRelatedResC(decodeBuffer2, callBackMessageNonCallRelatedRes);
            stat = callBackMessageNonCallRelatedResC.Decode();
            if (stat == 0) {
               /* process the callBackMessageNonCallRelated operation in your application */
               if (tracearg) {
                  printf ("Decode of CallBackMessageNonCallRelatedResult was successful\n");
                  callBackMessageNonCallRelatedResC.Print ("CallBackMessageNonCallRelatedResult");
               }
            }
            else {
               printf ("Decode of CallBackMessageNonCallRelatedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 302 : {  /* cancelCallBack operation */
            ASN1T_CancelCallBackResult cancelCallBackRes;
            ASN1C_CancelCallBackResult cancelCallBackResC(decodeBuffer2, cancelCallBackRes);
            stat = cancelCallBackResC.Decode();
            if (stat == 0) {
               /* process the cancelCallBack operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelCallBackResult was successful\n");
                  cancelCallBackResC.Print ("CancelCallBackResult");
               }
            }
            else {
               printf ("Decode of CancelCallBackResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 303 : {  /* cancelCallBackMessage operation */
            ASN1T_CancelCallBackMessageResult cancelCallBackMessageRes;
            ASN1C_CancelCallBackMessageResult cancelCallBackMessageResC(decodeBuffer2, cancelCallBackMessageRes);
            stat = cancelCallBackMessageResC.Decode();
            if (stat == 0) {
               /* process the cancelCallBackMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of CancelCallBackMessageResult was successful\n");
                  cancelCallBackMessageResC.Print ("CancelCallBackMessageResult");
               }
            }
            else {
               printf ("Decode of CancelCallBackMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 304 : {  /* getAgentState operation */
            ASN1T_GetAgentStateResult getAgentStateRes;
            ASN1C_GetAgentStateResult getAgentStateResC(decodeBuffer2, getAgentStateRes);
            stat = getAgentStateResC.Decode();
            if (stat == 0) {
               /* process the getAgentState operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAgentStateResult was successful\n");
                  getAgentStateResC.Print ("GetAgentStateResult");
               }
            }
            else {
               printf ("Decode of GetAgentStateResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 305 : {  /* getAutoAnswer operation */
            ASN1T_GetAutoAnswerResult getAutoAnswerRes;
            ASN1C_GetAutoAnswerResult getAutoAnswerResC(decodeBuffer2, getAutoAnswerRes);
            stat = getAutoAnswerResC.Decode();
            if (stat == 0) {
               /* process the getAutoAnswer operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAutoAnswerResult was successful\n");
                  getAutoAnswerResC.Print ("GetAutoAnswerResult");
               }
            }
            else {
               printf ("Decode of GetAutoAnswerResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 306 : {  /* getAutoWorkMode operation */
            ASN1T_GetAutoWorkModeResult getAutoWorkModeRes;
            ASN1C_GetAutoWorkModeResult getAutoWorkModeResC(decodeBuffer2, getAutoWorkModeRes);
            stat = getAutoWorkModeResC.Decode();
            if (stat == 0) {
               /* process the getAutoWorkMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetAutoWorkModeResult was successful\n");
                  getAutoWorkModeResC.Print ("GetAutoWorkModeResult");
               }
            }
            else {
               printf ("Decode of GetAutoWorkModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 307 : {  /* getCallerIDStatus operation */
            ASN1T_GetCallerIDStatusResult getCallerIDStatusRes;
            ASN1C_GetCallerIDStatusResult getCallerIDStatusResC(decodeBuffer2, getCallerIDStatusRes);
            stat = getCallerIDStatusResC.Decode();
            if (stat == 0) {
               /* process the getCallerIDStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of GetCallerIDStatusResult was successful\n");
                  getCallerIDStatusResC.Print ("GetCallerIDStatusResult");
               }
            }
            else {
               printf ("Decode of GetCallerIDStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 308 : {  /* getDoNotDisturb operation */
            ASN1T_GetDoNotDisturbResult getDoNotDisturbRes;
            ASN1C_GetDoNotDisturbResult getDoNotDisturbResC(decodeBuffer2, getDoNotDisturbRes);
            stat = getDoNotDisturbResC.Decode();
            if (stat == 0) {
               /* process the getDoNotDisturb operation in your application */
               if (tracearg) {
                  printf ("Decode of GetDoNotDisturbResult was successful\n");
                  getDoNotDisturbResC.Print ("GetDoNotDisturbResult");
               }
            }
            else {
               printf ("Decode of GetDoNotDisturbResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 309 : {  /* getForwarding operation */
            ASN1T_GetForwardingResult getForwardingRes;
            ASN1C_GetForwardingResult getForwardingResC(decodeBuffer2, getForwardingRes);
            stat = getForwardingResC.Decode();
            if (stat == 0) {
               /* process the getForwarding operation in your application */
               if (tracearg) {
                  printf ("Decode of GetForwardingResult was successful\n");
                  getForwardingResC.Print ("GetForwardingResult");
               }
            }
            else {
               printf ("Decode of GetForwardingResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 310 : {  /* getLastNumberDialed operation */
            ASN1T_GetLastNumberDialedResult getLastNumberDialedRes;
            ASN1C_GetLastNumberDialedResult getLastNumberDialedResC(decodeBuffer2, getLastNumberDialedRes);
            stat = getLastNumberDialedResC.Decode();
            if (stat == 0) {
               /* process the getLastNumberDialed operation in your application */
               if (tracearg) {
                  printf ("Decode of GetLastNumberDialedResult was successful\n");
                  getLastNumberDialedResC.Print ("GetLastNumberDialedResult");
               }
            }
            else {
               printf ("Decode of GetLastNumberDialedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 311 : {  /* getRouteingMode operation */
            ASN1T_GetRouteingModeResult getRouteingModeRes;
            ASN1C_GetRouteingModeResult getRouteingModeResC(decodeBuffer2, getRouteingModeRes);
            stat = getRouteingModeResC.Decode();
            if (stat == 0) {
               /* process the getRouteingMode operation in your application */
               if (tracearg) {
                  printf ("Decode of GetRouteingModeResult was successful\n");
                  getRouteingModeResC.Print ("GetRouteingModeResult");
               }
            }
            else {
               printf ("Decode of GetRouteingModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 312 : {  /* setAgentState operation */
            ASN1T_SetAgentStateResult setAgentStateRes;
            ASN1C_SetAgentStateResult setAgentStateResC(decodeBuffer2, setAgentStateRes);
            stat = setAgentStateResC.Decode();
            if (stat == 0) {
               /* process the setAgentState operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAgentStateResult was successful\n");
                  setAgentStateResC.Print ("SetAgentStateResult");
               }
            }
            else {
               printf ("Decode of SetAgentStateResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 313 : {  /* setAutoAnswer operation */
            ASN1T_SetAutoAnswerResult setAutoAnswerRes;
            ASN1C_SetAutoAnswerResult setAutoAnswerResC(decodeBuffer2, setAutoAnswerRes);
            stat = setAutoAnswerResC.Decode();
            if (stat == 0) {
               /* process the setAutoAnswer operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAutoAnswerResult was successful\n");
                  setAutoAnswerResC.Print ("SetAutoAnswerResult");
               }
            }
            else {
               printf ("Decode of SetAutoAnswerResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 314 : {  /* setAutoWorkMode operation */
            ASN1T_SetAutoWorkModeResult setAutoWorkModeRes;
            ASN1C_SetAutoWorkModeResult setAutoWorkModeResC(decodeBuffer2, setAutoWorkModeRes);
            stat = setAutoWorkModeResC.Decode();
            if (stat == 0) {
               /* process the setAutoWorkMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetAutoWorkModeResult was successful\n");
                  setAutoWorkModeResC.Print ("SetAutoWorkModeResult");
               }
            }
            else {
               printf ("Decode of SetAutoWorkModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 315 : {  /* setCallerIDStatus operation */
            ASN1T_SetCallerIDStatusResult setCallerIDStatusRes;
            ASN1C_SetCallerIDStatusResult setCallerIDStatusResC(decodeBuffer2, setCallerIDStatusRes);
            stat = setCallerIDStatusResC.Decode();
            if (stat == 0) {
               /* process the setCallerIDStatus operation in your application */
               if (tracearg) {
                  printf ("Decode of SetCallerIDStatusResult was successful\n");
                  setCallerIDStatusResC.Print ("SetCallerIDStatusResult");
               }
            }
            else {
               printf ("Decode of SetCallerIDStatusResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 316 : {  /* setDoNotDisturb operation */
            ASN1T_SetDoNotDisturbResult setDoNotDisturbRes;
            ASN1C_SetDoNotDisturbResult setDoNotDisturbResC(decodeBuffer2, setDoNotDisturbRes);
            stat = setDoNotDisturbResC.Decode();
            if (stat == 0) {
               /* process the setDoNotDisturb operation in your application */
               if (tracearg) {
                  printf ("Decode of SetDoNotDisturbResult was successful\n");
                  setDoNotDisturbResC.Print ("SetDoNotDisturbResult");
               }
            }
            else {
               printf ("Decode of SetDoNotDisturbResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 317 : {  /* setForwarding operation */
            ASN1T_SetForwardingResult setForwardingRes;
            ASN1C_SetForwardingResult setForwardingResC(decodeBuffer2, setForwardingRes);
            stat = setForwardingResC.Decode();
            if (stat == 0) {
               /* process the setForwarding operation in your application */
               if (tracearg) {
                  printf ("Decode of SetForwardingResult was successful\n");
                  setForwardingResC.Print ("SetForwardingResult");
               }
            }
            else {
               printf ("Decode of SetForwardingResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 318 : {  /* setRouteingMode operation */
            ASN1T_SetRouteingModeResult setRouteingModeRes;
            ASN1C_SetRouteingModeResult setRouteingModeResC(decodeBuffer2, setRouteingModeRes);
            stat = setRouteingModeResC.Decode();
            if (stat == 0) {
               /* process the setRouteingMode operation in your application */
               if (tracearg) {
                  printf ("Decode of SetRouteingModeResult was successful\n");
                  setRouteingModeResC.Print ("SetRouteingModeResult");
               }
            }
            else {
               printf ("Decode of SetRouteingModeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 340 : {  /* ioRegister operation */
            ASN1T_IORegisterResult ioRegisterRes;
            ASN1C_IORegisterResult ioRegisterResC(decodeBuffer2, ioRegisterRes);
            stat = ioRegisterResC.Decode();
            if (stat == 0) {
               /* process the ioRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of IORegisterResult was successful\n");
                  ioRegisterResC.Print ("IORegisterResult");
               }
            }
            else {
               printf ("Decode of IORegisterResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 342 : {  /* ioRegisterCancel operation */
            ASN1T_IORegisterCancelResult ioRegisterCancelRes;
            ASN1C_IORegisterCancelResult ioRegisterCancelResC(decodeBuffer2, ioRegisterCancelRes);
            stat = ioRegisterCancelResC.Decode();
            if (stat == 0) {
               /* process the ioRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of IORegisterCancelResult was successful\n");
                  ioRegisterCancelResC.Print ("IORegisterCancelResult");
               }
            }
            else {
               printf ("Decode of IORegisterCancelResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 343 : {  /* dataCollected operation */
            ASN1T_DataCollectedResult dataCollectedRes;
            ASN1C_DataCollectedResult dataCollectedResC(decodeBuffer2, dataCollectedRes);
            stat = dataCollectedResC.Decode();
            if (stat == 0) {
               /* process the dataCollected operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectedResult was successful\n");
                  dataCollectedResC.Print ("DataCollectedResult");
               }
            }
            else {
               printf ("Decode of DataCollectedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 344 : {  /* dataCollectionResumed operation */
            ASN1T_DataCollectionResumedResult dataCollectionResumedRes;
            ASN1C_DataCollectionResumedResult dataCollectionResumedResC(decodeBuffer2, dataCollectionResumedRes);
            stat = dataCollectionResumedResC.Decode();
            if (stat == 0) {
               /* process the dataCollectionResumed operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectionResumedResult was successful\n");
                  dataCollectionResumedResC.Print ("DataCollectionResumedResult");
               }
            }
            else {
               printf ("Decode of DataCollectionResumedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 345 : {  /* dataCollectionSuspended operation */
            ASN1T_DataCollectionSuspendedResult dataCollectionSuspendedRes;
            ASN1C_DataCollectionSuspendedResult dataCollectionSuspendedResC(decodeBuffer2, dataCollectionSuspendedRes);
            stat = dataCollectionSuspendedResC.Decode();
            if (stat == 0) {
               /* process the dataCollectionSuspended operation in your application */
               if (tracearg) {
                  printf ("Decode of DataCollectionSuspendedResult was successful\n");
                  dataCollectionSuspendedResC.Print ("DataCollectionSuspendedResult");
               }
            }
            else {
               printf ("Decode of DataCollectionSuspendedResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 346 : {  /* resumeDataCollection operation */
            ASN1T_ResumeDataCollectionResult resumeDataCollectionRes;
            ASN1C_ResumeDataCollectionResult resumeDataCollectionResC(decodeBuffer2, resumeDataCollectionRes);
            stat = resumeDataCollectionResC.Decode();
            if (stat == 0) {
               /* process the resumeDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeDataCollectionResult was successful\n");
                  resumeDataCollectionResC.Print ("ResumeDataCollectionResult");
               }
            }
            else {
               printf ("Decode of ResumeDataCollectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 347 : {  /* startDataCollection operation */
            ASN1T_StartDataCollectionResult startDataCollectionRes;
            ASN1C_StartDataCollectionResult startDataCollectionResC(decodeBuffer2, startDataCollectionRes);
            stat = startDataCollectionResC.Decode();
            if (stat == 0) {
               /* process the startDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of StartDataCollectionResult was successful\n");
                  startDataCollectionResC.Print ("StartDataCollectionResult");
               }
            }
            else {
               printf ("Decode of StartDataCollectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 348 : {  /* stopDataCollection operation */
            ASN1T_StopDataCollectionResult stopDataCollectionRes;
            ASN1C_StopDataCollectionResult stopDataCollectionResC(decodeBuffer2, stopDataCollectionRes);
            stat = stopDataCollectionResC.Decode();
            if (stat == 0) {
               /* process the stopDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of StopDataCollectionResult was successful\n");
                  stopDataCollectionResC.Print ("StopDataCollectionResult");
               }
            }
            else {
               printf ("Decode of StopDataCollectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 349 : {  /* suspendDataCollection operation */
            ASN1T_SuspendDataCollectionResult suspendDataCollectionRes;
            ASN1C_SuspendDataCollectionResult suspendDataCollectionResC(decodeBuffer2, suspendDataCollectionRes);
            stat = suspendDataCollectionResC.Decode();
            if (stat == 0) {
               /* process the suspendDataCollection operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendDataCollectionResult was successful\n");
                  suspendDataCollectionResC.Print ("SuspendDataCollectionResult");
               }
            }
            else {
               printf ("Decode of SuspendDataCollectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 360 : {  /* cDRNotification operation */
            ASN1T_CDRNotificationResult cDRNotificationRes;
            ASN1C_CDRNotificationResult cDRNotificationResC(decodeBuffer2, cDRNotificationRes);
            stat = cDRNotificationResC.Decode();
            if (stat == 0) {
               /* process the cDRNotification operation in your application */
               if (tracearg) {
                  printf ("Decode of CDRNotificationResult was successful\n");
                  cDRNotificationResC.Print ("CDRNotificationResult");
               }
            }
            else {
               printf ("Decode of CDRNotificationResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 361 : {  /* cDRReport operation */
            ASN1T_CDRReportResult cDRReportRes;
            ASN1C_CDRReportResult cDRReportResC(decodeBuffer2, cDRReportRes);
            stat = cDRReportResC.Decode();
            if (stat == 0) {
               /* process the cDRReport operation in your application */
               if (tracearg) {
                  printf ("Decode of CDRReportResult was successful\n");
                  cDRReportResC.Print ("CDRReportResult");
               }
            }
            else {
               printf ("Decode of CDRReportResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 362 : {  /* sendStoredCDR operation */
            ASN1T_SendStoredCDRResult sendStoredCDRRes;
            ASN1C_SendStoredCDRResult sendStoredCDRResC(decodeBuffer2, sendStoredCDRRes);
            stat = sendStoredCDRResC.Decode();
            if (stat == 0) {
               /* process the sendStoredCDR operation in your application */
               if (tracearg) {
                  printf ("Decode of SendStoredCDRResult was successful\n");
                  sendStoredCDRResC.Print ("SendStoredCDRResult");
               }
            }
            else {
               printf ("Decode of SendStoredCDRResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 363 : {  /* startCDRTransmission operation */
            ASN1T_StartCDRTransmissionResult startCDRTransmissionRes;
            ASN1C_StartCDRTransmissionResult startCDRTransmissionResC(decodeBuffer2, startCDRTransmissionRes);
            stat = startCDRTransmissionResC.Decode();
            if (stat == 0) {
               /* process the startCDRTransmission operation in your application */
               if (tracearg) {
                  printf ("Decode of StartCDRTransmissionResult was successful\n");
                  startCDRTransmissionResC.Print ("StartCDRTransmissionResult");
               }
            }
            else {
               printf ("Decode of StartCDRTransmissionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 364 : {  /* stopCDRTransmission operation */
            ASN1T_StopCDRTransmissionResult stopCDRTransmissionRes;
            ASN1C_StopCDRTransmissionResult stopCDRTransmissionResC(decodeBuffer2, stopCDRTransmissionRes);
            stat = stopCDRTransmissionResC.Decode();
            if (stat == 0) {
               /* process the stopCDRTransmission operation in your application */
               if (tracearg) {
                  printf ("Decode of StopCDRTransmissionResult was successful\n");
                  stopCDRTransmissionResC.Print ("StopCDRTransmissionResult");
               }
            }
            else {
               printf ("Decode of StopCDRTransmissionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 365 : {  /* escapeRegister operation */
            ASN1T_EscapeRegisterResult escapeRegisterRes;
            ASN1C_EscapeRegisterResult escapeRegisterResC(decodeBuffer2, escapeRegisterRes);
            stat = escapeRegisterResC.Decode();
            if (stat == 0) {
               /* process the escapeRegister operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeRegisterResult was successful\n");
                  escapeRegisterResC.Print ("EscapeRegisterResult");
               }
            }
            else {
               printf ("Decode of EscapeRegisterResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 367 : {  /* escapeRegisterCancel operation */
            ASN1T_EscapeRegisterCancelResult escapeRegisterCancelRes;
            ASN1C_EscapeRegisterCancelResult escapeRegisterCancelResC(decodeBuffer2, escapeRegisterCancelRes);
            stat = escapeRegisterCancelResC.Decode();
            if (stat == 0) {
               /* process the escapeRegisterCancel operation in your application */
               if (tracearg) {
                  printf ("Decode of EscapeRegisterCancelResult was successful\n");
                  escapeRegisterCancelResC.Print ("EscapeRegisterCancelResult");
               }
            }
            else {
               printf ("Decode of EscapeRegisterCancelResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 368 : {  /* privateDataVersionSelection operation */
            ASN1T_PrivateDataVersionSelectionResult privateDataVersionSelectionRes;
            ASN1C_PrivateDataVersionSelectionResult privateDataVersionSelectionResC(decodeBuffer2, privateDataVersionSelectionRes);
            stat = privateDataVersionSelectionResC.Decode();
            if (stat == 0) {
               /* process the privateDataVersionSelection operation in your application */
               if (tracearg) {
                  printf ("Decode of PrivateDataVersionSelectionResult was successful\n");
                  privateDataVersionSelectionResC.Print ("PrivateDataVersionSelectionResult");
               }
            }
            else {
               printf ("Decode of PrivateDataVersionSelectionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 500 : {  /* concatenateMessage operation */
            ASN1T_ConcatenateMessageResult concatenateMessageRes;
            ASN1C_ConcatenateMessageResult concatenateMessageResC(decodeBuffer2, concatenateMessageRes);
            stat = concatenateMessageResC.Decode();
            if (stat == 0) {
               /* process the concatenateMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of ConcatenateMessageResult was successful\n");
                  concatenateMessageResC.Print ("ConcatenateMessageResult");
               }
            }
            else {
               printf ("Decode of ConcatenateMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 501 : {  /* deleteMessage operation */
            ASN1T_DeleteMessageResult deleteMessageRes;
            ASN1C_DeleteMessageResult deleteMessageResC(decodeBuffer2, deleteMessageRes);
            stat = deleteMessageResC.Decode();
            if (stat == 0) {
               /* process the deleteMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of DeleteMessageResult was successful\n");
                  deleteMessageResC.Print ("DeleteMessageResult");
               }
            }
            else {
               printf ("Decode of DeleteMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 502 : {  /* playMessage operation */
            ASN1T_PlayMessageResult playMessageRes;
            ASN1C_PlayMessageResult playMessageResC(decodeBuffer2, playMessageRes);
            stat = playMessageResC.Decode();
            if (stat == 0) {
               /* process the playMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of PlayMessageResult was successful\n");
                  playMessageResC.Print ("PlayMessageResult");
               }
            }
            else {
               printf ("Decode of PlayMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 503 : {  /* queryVoiceAttribute operation */
            ASN1T_QueryVoiceAttributeResult queryVoiceAttributeRes;
            ASN1C_QueryVoiceAttributeResult queryVoiceAttributeResC(decodeBuffer2, queryVoiceAttributeRes);
            stat = queryVoiceAttributeResC.Decode();
            if (stat == 0) {
               /* process the queryVoiceAttribute operation in your application */
               if (tracearg) {
                  printf ("Decode of QueryVoiceAttributeResult was successful\n");
                  queryVoiceAttributeResC.Print ("QueryVoiceAttributeResult");
               }
            }
            else {
               printf ("Decode of QueryVoiceAttributeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 511 : {  /* recordMessage operation */
            ASN1T_RecordMessageResult recordMessageRes;
            ASN1C_RecordMessageResult recordMessageResC(decodeBuffer2, recordMessageRes);
            stat = recordMessageResC.Decode();
            if (stat == 0) {
               /* process the recordMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of RecordMessageResult was successful\n");
                  recordMessageResC.Print ("RecordMessageResult");
               }
            }
            else {
               printf ("Decode of RecordMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 504 : {  /* reposition operation */
            ASN1T_RepositionResult repositionRes;
            ASN1C_RepositionResult repositionResC(decodeBuffer2, repositionRes);
            stat = repositionResC.Decode();
            if (stat == 0) {
               /* process the reposition operation in your application */
               if (tracearg) {
                  printf ("Decode of RepositionResult was successful\n");
                  repositionResC.Print ("RepositionResult");
               }
            }
            else {
               printf ("Decode of RepositionResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 505 : {  /* resume operation */
            ASN1T_ResumeResult resumeRes;
            ASN1C_ResumeResult resumeResC(decodeBuffer2, resumeRes);
            stat = resumeResC.Decode();
            if (stat == 0) {
               /* process the resume operation in your application */
               if (tracearg) {
                  printf ("Decode of ResumeResult was successful\n");
                  resumeResC.Print ("ResumeResult");
               }
            }
            else {
               printf ("Decode of ResumeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 506 : {  /* review operation */
            ASN1T_ReviewResult reviewRes;
            ASN1C_ReviewResult reviewResC(decodeBuffer2, reviewRes);
            stat = reviewResC.Decode();
            if (stat == 0) {
               /* process the review operation in your application */
               if (tracearg) {
                  printf ("Decode of ReviewResult was successful\n");
                  reviewResC.Print ("ReviewResult");
               }
            }
            else {
               printf ("Decode of ReviewResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 507 : {  /* setVoiceAttribute operation */
            ASN1T_SetVoiceAttributeResult setVoiceAttributeRes;
            ASN1C_SetVoiceAttributeResult setVoiceAttributeResC(decodeBuffer2, setVoiceAttributeRes);
            stat = setVoiceAttributeResC.Decode();
            if (stat == 0) {
               /* process the setVoiceAttribute operation in your application */
               if (tracearg) {
                  printf ("Decode of SetVoiceAttributeResult was successful\n");
                  setVoiceAttributeResC.Print ("SetVoiceAttributeResult");
               }
            }
            else {
               printf ("Decode of SetVoiceAttributeResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 508 : {  /* stop operation */
            ASN1T_StopResult stopRes;
            ASN1C_StopResult stopResC(decodeBuffer2, stopRes);
            stat = stopResC.Decode();
            if (stat == 0) {
               /* process the stop operation in your application */
               if (tracearg) {
                  printf ("Decode of StopResult was successful\n");
                  stopResC.Print ("StopResult");
               }
            }
            else {
               printf ("Decode of StopResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 509 : {  /* suspend operation */
            ASN1T_SuspendResult suspendRes;
            ASN1C_SuspendResult suspendResC(decodeBuffer2, suspendRes);
            stat = suspendResC.Decode();
            if (stat == 0) {
               /* process the suspend operation in your application */
               if (tracearg) {
                  printf ("Decode of SuspendResult was successful\n");
                  suspendResC.Print ("SuspendResult");
               }
            }
            else {
               printf ("Decode of SuspendResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return ;
            }
         } break;

      case 510 : {  /* synthesizeMessage operation */
            ASN1T_SynthesizeMessageResult synthesizeMessageRes;
            ASN1C_SynthesizeMessageResult synthesizeMessageResC(decodeBuffer2, synthesizeMessageRes);
            stat = synthesizeMessageResC.Decode();
            if (stat == 0) {
               /* process the synthesizeMessage operation in your application */
               if (tracearg) {
                  printf ("Decode of SynthesizeMessageResult was successful\n");
                  synthesizeMessageResC.Print ("SynthesizeMessageResult");
               }
            }
            else {
               printf ("Decode of SynthesizeMessageResult failed\n");
               decodeBuffer2.printErrorInfo ();
               return;
            }
         } break;

      default: { /* error: invalid operation code, not defined in csta phase 3*/
         printf ("error: invalid operation code, not defined in csta phase 3\n");
         } break;
   }
}



void Client::processReturnError(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL tracearg)
{
   /* 
   return error processing will require to decode the ASN1T_ReturnError.parameter
   opentype in to real error type, only one possible error type can be returned
   in csta 3, which is "UniversalFailure" 
   
   ASN1T_ReturnError.errcode field is used to findout the real type for 
   ASN1T_ReturnError.parameter(opentype). Real type for all the operation
   in Table 1(i.e csta3) is "UniversalFailure"
   */
   int stat;
      ASN1BERDecodeBuffer decodeBuffer2 
         (pdu.u.returnError->parameter.data, pdu.u.returnError->parameter.numocts);

      ASN1T_UniversalFailure uniFailure;
      ASN1C_UniversalFailure uniFailureC (decodeBuffer2, uniFailure);

      stat = uniFailureC.Decode ();

      if (stat == 0) {
         /* process the ReturnError in your application */

         if (tracearg) {
            printf ("Decode of ReturnError was successful\n");
            uniFailureC.Print ("UniversalFailure");
         }
      }
      else {
         printf ("Decode of ReturnError failed\n");
         decodeBuffer2.printErrorInfo ();
         return ;
      }
}



void Client::processReject(ASN1T_CSTA_ROSE_PDU& pdu, OSBOOL tracearg)
{
   /* process the reject in your application */
}

